/**
   The static method getPrefix() is a recursive descent parser for
   strings that hold the prefix notation of an arithmetic expression.

   Here is the BNF grammar for prefix notation expressions.

      Prefix -> BinOp Prefix Prefix
              | UnOp Prefix
              | Number

       BinOp -> '+' | '-' | '*' | '/'

       UnOp  -> 'neg' | 'sqrt'

      Number -> any valid Java double

   In a recursive descent parser, every production is handled by a method.
   If a production is recursive, then the method will be recursive.
   If two productions are mutually recursive, then the two methods will
   be mutually recursive. Since this grammar has four productions, the
   parser consists of four methods, getPrefix(), getBinOP(), getUnOp(),
   and getNumber().

   This parser creates a parse tree (that is, a "syntax tree")
   for each prefix expression.
*/

public class ParseExpr0ParseTree
{
   public static Tree parseExpr0(String expr)
   {
      Tokenizer tokens = new Tokenizer(expr);

      Tree result = getPrefix(tokens); // parse the token stream

      if ( tokens.hasToken() )  // there shouldn't be any more tokens
         throw new Error("syntax error: unexpected input: "+tokens.peekToken()+"\n"+tokens);

      return result;
   }//parseExpr0()



   private static Tree getPrefix(Tokenizer tokens)
   {
      Tree result = new Tree( "Prefix" ); // node that represents the grammar nonterminal

      String token = tokens.peekToken();  // one token look ahead

      if ( "+-*/^".indexOf(token) != -1 ) // binary operator case
      {
         result.addSubTree( getBinOp(tokens) );  // subtree for the operator token
         result.addSubTree( getPrefix(tokens) ); // subtree for the first operand
         result.addSubTree( getPrefix(tokens) ); // subtree for the second operand
      }
      else if ( "neg sqrt".indexOf(token) != -1 ) // unary operator case
      {
         result.addSubTree( getUnOp(tokens) );   // subtree for the operator token
         result.addSubTree( getPrefix(tokens) ); // subtree for the (only) operand
      }
      else // operand case (this case terminates the recursion)
      {
         result.addSubTree( getNumber(tokens) );
      }
      return result;
   }//getPrefix()


   private static Tree getBinOp(Tokenizer tokens)
   {
      Tree result = new Tree( "BinOp" );

      String token = tokens.nextToken(); // consume one token

      result.addSubTree( new Tree(token) ); // subtree for the operator token

      return result;
   }//getBinOp()


   private static Tree getUnOp(Tokenizer tokens)
   {
      Tree result = new Tree( "UnOp" );

      String token = tokens.nextToken(); // consume one token

      result.addSubTree( new Tree(token) ); // subtree for the operator token

      return result;
   }//getUnOp()


   private static Tree getNumber(Tokenizer tokens)
   {
      Tree result = new Tree( "Number" );

      String token = tokens.nextToken(); // consume one token

      result.addSubTree( new Tree(token) ); // subtree for the number token

      return result;
   }//getNumber()

}//ParsePrefix2ParseTree
