/*
   The first few Fibonacci numbers are

   0, 1, 1, 2, 3, 5, 8. 13, 21, 34, 55, 89, 144, ...

   If we let F(n) denote the n'th Fibonacci number, then these
   numbers are defined by the following recurrence relation.

   F(0) = 0
   F(1) = 1
   F(n) = F(n-1) + F(n-2) for n >= 2

   You can find a list of Fibonacci numbers at
      http://planetmath.org/encyclopedia/ListOfFibonacciNumbers.html

   Below are several ways to compute F(n), the n'th Fibonacci number.

   If you are using a 32 computer (or compiler?) you can only
   compute up to Fib(47)=2971215073.
*/


/*
  Here is the most basic way to compute the n'th
  Fibonacci number. We directly implement the
  recurrence relation as a recursive function.
*/
unsigned long Fib1(int n)
{
   unsigned long fn;
   if ( n == 0 )
      fn = 0;
   else if ( n == 1 )
      fn = 1;
   else
      fn = Fib1(n-1) + Fib1(n-2);
   return fn;
}



/*
   Here is a "bottom up dynamic programming"
   implementation of the n'th Fibonacci number.
*/
unsigned long Fib2(int n)
{
   int i;
   unsigned long fn;
   unsigned long* dpTable = malloc((n+1)*sizeof(long));
   if (dpTable == NULL)  // error check
   {
      fprintf(stderr, "Fib2: not enough memory for array of size %d.\n", n);
      return -1;
   }
   dpTable[0] = 0;
   dpTable[1] = 1;
   for (i = 2; i <= n; i++)
   {
      dpTable[i] = dpTable[i-1] + dpTable[i-2];
   }
   fn = dpTable[n];
   free(dpTable);
   return fn;
}



/*
   Here is another "bottom up dynamic programming"
   implementation of the n'th Fibonacci number.
   This version does not need an n-element array
   to compute Fib3(n). This version just needs
   three variables.
*/
unsigned long Fib3(int n)
{
   int i;
   unsigned long fi, fi_minus_1, fi_minus_2;

   if ( n == 0 )
      fi = 0;
   else if ( n == 1 )
      fi = 1;
   else
   {
      fi_minus_2 = 0;
      fi_minus_1 = 1;
      for (i = 2; i <= n; i++)
      {
         fi = fi_minus_1 + fi_minus_2;
         fi_minus_2 = fi_minus_1;
         fi_minus_1 = fi;
      }
   }
   return fi;
}



/*
   Here is a "top down dynamic programming"
   implementation of the n'th Fibonacci number.
   This is also called "memoization dynamic programing".
*/
unsigned long Fib4Helper(int n, unsigned long* dpTable);

unsigned long Fib4(int n)
{
   int i;
   unsigned long fn;
   unsigned long* dpTable = malloc((n+1)*sizeof(long));
   if (dpTable == NULL)  // error check
   {
      fprintf(stderr, "Fib4: not enough memory for array of size %d.\n", n);
      return -1;
   }

   for (i = 0; i <= n; i++)
      dpTable[i] = -1;  // initialize the table with "null" values

   fn = Fib4Helper(n, dpTable);
   free(dpTable);
   return fn;
}

unsigned long Fib4Helper(int n, unsigned long* dpTable)
{
   unsigned long fn;
   if ( n == 0 )
      fn = 0;
   else if ( n == 1 )
      fn = 1;
   else  // Compare this with the purely recursive verion, Fib1().
   {     // This is kind of a "short circuited" recursion.
      if ( dpTable[n-1] == -1 ) dpTable[n-1] = Fib4Helper(n-1, dpTable);
      if ( dpTable[n-2] == -1 ) dpTable[n-2] = Fib4Helper(n-2, dpTable);
      fn = dpTable[n-1] + dpTable[n-2];
   }
   return fn;
}



/*
  Here is a slightly different way to write Fib4Helper().
  This does the memoization in a slightly different way.
  It "reverses" the order of the recursion and the memoization.
*/
unsigned long Fib4Helper_2(int n, unsigned long* dpTable)
{
   unsigned long fn;
   if ( dpTable[n] != -1 ) return dpTable[n]; // Look in the table first.
   if ( n == 0 )                              // If the result is not in the table,
      fn = 0;                                 // then do the recursion.
   else if ( n == 1 )
      fn = 1;
   else
      fn = Fib4Helper_2(n-1, dpTable) + Fib4Helper_2(n-2, dpTable);
   dpTable[n] = fn; // Store the new result in the table before returning.
   return fn;
}
