/*
   Roger Kraft
   Associate Professor
   Department of Mathematics, Computer Science, and Statistics
   Purdue University Calumet
   roger@purduecal.edu
   http://cs.purduecal.edu/~rlkraft/


   Here is an outline of how to use the functions and declarations
   from this header to time some operation taking place in the
   current thread.

      // declare three TimingObject objects
      TimingObject startTime, endTime, elapsedTime;

      // begin timing an operation
      getThreadTime( &startTime );

         // do some operation that needs to be timed ...

      // stop timing the operation
      getThreadTime( &endTime );

      // compute the elapsed time
      computeElapsedTime( &startTime, &endTime, &elapsedTime );

      // print the result
      printTimingObject( &elapsedTime );

   In a similar way, you can time operations taking place in
   the current process by calling getProcessTime() in place of
   getThreadTimes() in the above outline.

   This timing information is not particularly accurate. It has a
   resolution of probably about 10 milliseconds. But one vary big
   advantage of these routines is that, in addition to the usual
   "wall clock time", they give both "user time" and "kernel time"
   (to about 10 msec). The "user time" measure is usually more
   useful than "wall clock time". So if you are timing an operation
   that takes more than a second or two, then "user time" will be
   much more reliable than "wall clock time".

   If you need higher resolution, use the Win32 functions
       QueryPerformanceFrequency()
       QueryPerformanceCounter()
   They will give you microsecond accurracy (for wall clock time).
   So if you are timing operations that are less than 10 milliseconds
   (but longer than a few microseconds), then use these two functions.

   This header file also contains two utility functions.

   The first
      void printProcessTimes(HANDLE hProcess)
   prints out the timing information for a completed process, as
   returned by GetProcessTimes(HANDLE hProcess, ...).

   The second
      void printThreadTimes(HANDLE hThread)
   prints out the timing information for a completed process, as
   returned by GetThreadTimes(HANDLE hThread, ...).
*/
#include <stdio.h>
#include <windows.h>


typedef struct
{
   SYSTEMTIME wallClockTime_ST;
   FILETIME userTime_FT;
   FILETIME kernelTime_FT;
   FILETIME junkTime_FT;
}TimingObject;


void getThreadTime(TimingObject* timingObject)
{
   HANDLE hThread;

   hThread = GetCurrentThread();

   GetSystemTime(&(timingObject -> wallClockTime_ST));
   GetThreadTimes(hThread, &(timingObject -> junkTime_FT),
                           &(timingObject -> junkTime_FT),
                           &(timingObject -> kernelTime_FT),
                           &(timingObject -> userTime_FT)
                 );
}//getThreadTime()


void getProcessTime(TimingObject* timingObject)
{
   HANDLE hProcess;

   hProcess = GetCurrentProcess();

   GetSystemTime(&(timingObject -> wallClockTime_ST));
   GetProcessTimes(hProcess, &(timingObject -> junkTime_FT),
                             &(timingObject -> junkTime_FT),
                             &(timingObject -> kernelTime_FT),
                             &(timingObject -> userTime_FT)
                 );
}//getProcessTime()


void computeElapsedTime(TimingObject* time1,       // in parameter
                        TimingObject* time2,       // in parameter
                        TimingObject* elapsedTime  // out parameter
                       )
{
   union  // structure used for file time arithmetic
   {      // See "Windows System Programming" 3rd Ed., J.M. Hart, page 184
      FILETIME ft;
      LONGLONG li;
   } startWCT, endWCT, elapsedWCT,
     startUT,  endUT,  elapsedUT,
     startKT,  endKT,  elapsedKT;

   // convert SYSTEMTIME to FILETIME so that it is easier to do time arithmetic
   SystemTimeToFileTime(&(time1 -> wallClockTime_ST), &startWCT.ft);
   SystemTimeToFileTime(&(time2 -> wallClockTime_ST), &endWCT.ft);

   startUT.ft = time1 -> userTime_FT;
   endUT.ft   = time2 -> userTime_FT;

   startKT.ft = time1 -> kernelTime_FT;
   endKT.ft   = time2 -> kernelTime_FT;

   // do all the time arithmetic
   elapsedWCT.li = endWCT.li - startWCT.li;
   elapsedUT.li  = endUT.li  - startUT.li;
   elapsedKT.li  = endKT.li  - startKT.li;

   // put all the results in the output parameter
   FileTimeToSystemTime(&elapsedWCT.ft, &(elapsedTime -> wallClockTime_ST));
   elapsedTime -> userTime_FT   = elapsedUT.ft;
   elapsedTime -> kernelTime_FT = elapsedKT.ft;

   return;
}//computeElapsedTime()



void printTimingObject(TimingObject* time)
{
   SYSTEMTIME wallClockTime, userTime, kernelTime;

   wallClockTime = time -> wallClockTime_ST;
   FileTimeToSystemTime(&(time -> userTime_FT),   &userTime);
   FileTimeToSystemTime(&(time -> kernelTime_FT), &kernelTime);

   // NOTE: The following printf() does not print out the wHour part of
   //       each SYSTEMTIME object, since most uses of this function will
   //       have that part equal to zero. It is easy to add code for wHour.
   printf("\tTime: %02d:%02d:%03dwct %02d:%02d:%03dut %02d:%02d:%03dkt\n",
            wallClockTime.wMinute, wallClockTime.wSecond, wallClockTime.wMilliseconds,
            userTime.wMinute, userTime.wSecond, userTime.wMilliseconds,
            kernelTime.wMinute, kernelTime.wSecond, kernelTime.wMilliseconds );
}//printTimingObject()



void printProcessTimes(HANDLE hProcess)
{
   union  // structure used for file time arithmetic
   {      // See "Windows System Programming" 3rd Ed., J.M. Hart, page 184
      FILETIME ft;
      LONGLONG li;
   } createTime, exitTime, elapsedTime;

   SYSTEMTIME elapsedTime_ST;
   FILETIME userTime_FT, kernelTime_FT;
   TimingObject to;

   GetProcessTimes(hProcess, &(createTime.ft), &(exitTime.ft),
                             &kernelTime_FT, &userTime_FT);
   elapsedTime.li = exitTime.li - createTime.li;
   FileTimeToSystemTime(&elapsedTime.ft, &elapsedTime_ST);
   to.wallClockTime_ST = elapsedTime_ST;
   to.userTime_FT = userTime_FT;
   to.kernelTime_FT = kernelTime_FT;
   printTimingObject(&to);
}//printProcessTimes()



void printThreadTimes(HANDLE hThread)
{
   union  // structure used for file time arithmetic
   {      // See "Windows System Programming" 3rd Ed., J.M. Hart, page 184
      FILETIME ft;
      LONGLONG li;
   } createTime, exitTime, elapsedTime;

   SYSTEMTIME elapsedTime_ST;
   FILETIME userTime_FT, kernelTime_FT;
   TimingObject to;

   GetThreadTimes(hThread, &(createTime.ft), &(exitTime.ft),
                           &kernelTime_FT, &userTime_FT);
   elapsedTime.li = exitTime.li - createTime.li;
   FileTimeToSystemTime(&elapsedTime.ft, &elapsedTime_ST);
   to.wallClockTime_ST = elapsedTime_ST;
   to.userTime_FT = userTime_FT;
   to.kernelTime_FT = kernelTime_FT;
   printTimingObject(&to);
}//printThreadTimes()
