/*
   This client implements a purely binary protocol.

   This client sends to the server a binary, four byte,
   integer value.

   Then this client expects back from the server that
   many binary, eight byte, doubles.

   Then this client closes its connection to the server.
*/

import java.net.*;
import java.io.*;
import java.util.Scanner;

public class NumericClient
{
   public static final int SERVER_PORT = 5000; // Should be above 1023.

   public static void main (String[] args)
   {
      Socket           socket = null;
      DataInputStream  in = null;
      DataOutputStream out = null;

      final String hostName;
      if (args.length > 0)
      {
         hostName = args[0];
      }
      else
      {
         hostName = "localhost";
      }

      final int portNumber;
      if (args.length > 1)
      {
         portNumber = Integer.parseInt(args[1]);
      }
      else
      {
         portNumber = SERVER_PORT;
      }

      // Get this client's process id number (PID). This helps
      // to identify the client in TaskManager or TCPView.
      final ProcessHandle handle = ProcessHandle.current();
      final long pid = handle.pid();
      System.out.println("CLIENT: Process ID number (PID): " + pid );

      // Make a connection to the server
      try
      {
         System.out.println("CLIENT: Connecting to server: " + hostName
                          + " on port " + portNumber );
         // Steps 4, 5, 6.
         socket = new Socket(InetAddress.getByName(hostName),
                             portNumber);

         in = new DataInputStream(               // Step 8.
                     socket.getInputStream());   // Step 7.

         out = new DataOutputStream(             // Step 8.
                      socket.getOutputStream()); // Step 7.
      }
      catch (IOException e)
      {
         System.out.println("CLIENT: Cannot connect to server.");
         e.printStackTrace();
         System.exit(-1);
      }
      System.out.println("CLIENT: Connected to server.");
      // Get this client's local port number and log it to the console.
      // This helps to identify this client in the server's transcript.
      final int port = socket.getLocalPort();
      System.out.println("CLIENT: Local Port: " + port);

      // Step 8.
      // Send the server a request.
      final Scanner stdin = new Scanner(System.in);
      System.out.print("How many doubles do you want from the server: ");
      final int request = stdin.nextInt();
      try
      {
         out.writeInt(request);
         out.flush(); // Make sure that the request is sent.
      }
      catch (IOException e)
      {
         System.out.println("CLIENT: Cannot send request to server.");
         //socket.close();  // should close the socket
         e.printStackTrace();
         System.exit(-1);
      }

      System.out.println("CLIENT: Request sent to the server.");

      // Receive the doubles from the server.
      for (int i = 0; i < request; ++i)
      {
         try
         {
            final double response = in.readDouble();
            // Log the double to the client's console.
            System.out.printf("CLIENT: Server response %d is: %f\n", i, response);
         }
         catch (IOException e)
         {
            System.out.println("CLIENT: Cannot receive response from server.");
            System.out.println( e );
         }
      }

      // Step 9.
      // Disconnect from the server.
      try
      {
         socket.close();
         System.out.println("CLIENT: Closed socket.");
      }
      catch (IOException e)
      {
         System.out.println("CLIENT: Cannot disconnect from server.");
         System.out.println( e );
      }
   }
}
