/*
   This example program is important because it gives us a simple example with which to analyze memory allocation in the rendering algorithms.

   A triangle is made up of three vertices and three line segments. But each line segment is made up of two vertices. So it would seem that when we model a triangle, we should have six vertices (not three), two for each line segment. How does the Java code resolve this?

   Let us draw a picture to see how the following Java code creates a Model object for a triangle.
         Model model = new Model();
         Vertex v1 = new Vertex(3.0, 0.0, 0.0);
         Vertex v2 = new Vertex(0.0, 3.0, 0.0);
         Vertex v3 = new Vertex(0.0, 0.0, 3.0);
         LineSegment ls1 = new LineSegment(v1, v2);
         LineSegment ls2 = new LineSegment(v2, v3);
         LineSegment ls3 = new LineSegment(v3, v1);
         model.lineSegmentList.add( ls1 );
         model.lineSegmentList.add( ls2 );
         model.lineSegmentList.add( ls3 );
   The Model object contains a list of references to three LineSegment objects. Each LineSegment object contains two references to Vertex objects. But there are only three Vertex objects, and they are "shared" by the LineSegment objects, as shown in the diagram below.

                  model
                    |
                    |
              LineSgementList
              /     |       \
             /      |         \
           ls1     ls2         ls3
    ...   /   \   /    \      /   \...
       \ /     \ /       \  /
        v1      v2        v3

   An important question that we need to answer is, does it make a difference that Vertex objects are shared by LineSegment objects? For example, notice that if v2 is ever mutated, then both ls1 and ls2 are also mutated. Is that OK? It turns out that sometimes it is OK, and sometimes it is not OK.

   Consider what happens to these data structures as they pass through the graphics rendering pipeline.

   The first stage of the renderer transforms vertices from world coordinates to view coordinates. Let us suppose that when this stage transforms a vertex, it mutates the Vertex object referenced by a LineSegment object (rather than create a new, transformed, Vertex object). Here is what the mutating code might look like when transforming Vertex v1 in LineSegment ls1.
         Vertex temp0 = viewMatrix.times(ls1.v[0]);
         ls1.v[0].x = temp0.x;  // now mutate ls1.v[0]
         ls1.v[0].y = temp0.y;
         ls1.v[0].z = temp0.z;
   In this stage, it is OK if the LineSegment objects share (the mutated) Vertex objects. In fact, it is better if the LineSgement objects share Vertex objects. Suppose we eliminated the shared vertices by making an extra copy of each vertex, so that there are two copies of each of v1, v2, and v3. Then the Model data structure would look something like this (how might you write the code to instantiate this model?).

                  model
                    |
                    |
              LineSgementList
              /     |       \
             /      |         \
           /        |           \
        ls1        ls2           ls3
       /   \      /   \         /   \
     v1    v2    v2   v3      v3    v1

   With this kind of structure, notice that we have twice as many Vertex objects that need to be transformed from world to view coordinates. If our model has thousands (or millions) of triangles, having twice as many vertices to transform can make quite a bit of difference in the pipeline's performance. So the sharing of vertices between line segments can save a lot of work in the transformation stage of the pipeline.


   Question: The transformation stage can create new Vertex objects, instead of mutating the existing Vertex objects referred to by the LineSegment objects. The code to transform Vertex v1 in LineSegment ls1 might look like this.
         ls1.v[0] = viewMatrix.times(ls.v[0]);
   (The times() method in the Matrix class returns a new object; look it up in the file Matrix.java.) If we start with the shared vertex data structure, what happens if the transformation stage creates new Vertex objects? Is this good or bad? How would you draw the Model data structure right after each Vertex object is transformed? How would you draw the Model data structure after all the Vertex objects are transformed?


   Now let us look ahead to the clipping stage of the pipeline. Suppose our triangle looks like the following picture, where vertex v2 needs to be clipped off the triangle. Let us suppose that when a LineSegment object gets clipped, the algorithm mutates the Vertex object that needs to be clipped from a LineSegment object. Suppose we are using the data structure with the shared vertices and suppose that the clipping algorithm decides to clip the line segment from v1 to v2 first.

                x=-1           x=1
                 |              |
            y=1 -+--------------+--
                 |       v1     |
                 |        +-----------+ v2
                 |        |     |   /
                 |        |     | /
                 |        |     /
                 |        |   / |
                 |        | /   |
                 |        +     |
                 |        v3    |
           y=-1 -+--------------+--
                 |              |

   Notice that the vertex to be clipped, v2, is shared by LineSegment(v1, v2) and LineSegment(v2, v3). When the line segment from v1 to v2 is clipped to the line segment from v1 to v2', by mutating v2 into v2', the line segment from v2 to v3 will also be clipped to the line segment from v2' to v3. The resulting triangle will look like the following, which is incorrect.

                x=-1           x=1
                 |              |
            y=1 -+--------------+--
                 |       v1     |
                 |        +-----+ v2'
                 |        |    /|
                 |        |   / |
                 |        |  /  |
                 |        | /   |
                 |        |/    |
                 |        +     |
                 |        v3    |
           y=-1 -+--------------+--
                 |              |

The model data structure, after the clipping, will look like this.

                  model
                    |
                    |
              LineSgementList
              /     |       \
             /      |         \
           ls1     ls2         ls3
    ...   /   \   /    \      /   \...
       \ /     \ /       \  /
        v1      v2'       v3

   On the other hand, if in the clipping algorithm we use the data structure that does not share Vertex objects, then the line segments from v1 to v2 and from v2 to v3 will be independent line segments and they will be clipped independently. Vertex v2 in LineSegment(v1, v2) will be mutated to vertex v2',

                x=-1           x=1
                 |              |
            y=1 -+--------------+--
                 |       v1     |
                 |        +-----+v2'  + v2
                 |        |     |   /
                 |        |     | /
                 |        |     /
                 |        |   / |
                 |        | /   |
                 |        +     |
                 |        v3    |
           y=-1 -+--------------+--
                 |              |

and then vertex v2 in LineSegment(v2, v3) will be mutated to vertex v2''.

                x=-1           x=1
                 |              |
            y=1 -+--------------+--
                 |       v1     |
                 |        +-----+ v2'
                 |        |     |
                 |        |     |
                 |        |     + v2''
                 |        |   / |
                 |        | /   |
                 |        +     |
                 |        v3    |
           y=-1 -+--------------+--
                 |              |

The model data structure, after the clipping, will look like this.

                  model
                    |
                    |
              LineSgementList
              /     |       \
             /      |         \
           /        |           \
        ls1        ls2           ls3
       /   \      /   \         /   \
     v1    v2'  v2''  v3      v3    v1

   From the above examples we see that it is sometimes good and sometimes bad to share vertices among line segments. The good part of sharing is better performance, but the bad part of sharing is an incorrect algorithm. The choice that we make in this renderer is to not share vertices among line segments. But we still want to write code like the code above for a triangle.
         Model model = new Model();
         Vertex v1 = new Vertex(3.0, 0.0, 0.0);
         Vertex v2 = new Vertex(0.0, 3.0, 0.0);
         Vertex v3 = new Vertex(0.0, 0.0, 3.0);
         LineSegment ls1 = new LineSegment(v1, v2);
         LineSegment ls2 = new LineSegment(v2, v3);
         LineSegment ls3 = new LineSegment(v3, v1);
         model.lineSegmentList.add( ls1 );
         model.lineSegmentList.add( ls2 );
         model.lineSegmentList.add( ls3 );
   This is the obvious and elegant code for instantiating a triangle. We don't want to mess it up with cumbersome code for creating redundant vertices. So how can we allow ourselves to write this code, but not end up with shared vertices and incorrect algorithms? The solution we use in this renderer is a "zeroth pipeline stage" that makes copies of every LineSegment and Vertex object in a Scene. Here is the code from this pipeline stage.

      public Model processModel(Model model)
      {  Model newModel = new Model(model); // copy everything except the geometry
         for (LineSegment ls : model.lineSegmentList)
         {
            LineSegment lsCopy = new LineSegment(ls);  // deep copy
            newModel.addLineSegment( lsCopy );
         }
         return newModel;
      }
   This code will convert the data structure on the left below to the one on the right, where every object represented on the right is a copy of the corresponding object on the left.

                model                                  newModel
                  |                                       |
                  |                                       |
            LineSgementList                         LineSgementList
            /     |       \                         /     |      \
           /      |         \                      /      |        \
         ls1     ls2         ls3                 /        |          \
  ...   /   \   /    \      /   \...          ls1        ls2          ls3
     \ /     \ /       \  /                  /   \      /   \        /   \
      v1      v2'       v3                 v1    v2    v2   v3     v3    v1

   In a later renderer we will solve this memory allocation problem in a more efficient way. We will do that by creating new "geometric primitives" and a new "primitive assembly" pipeline stage.
*/

public class MainTriangle
{
   public static void main(String[] args)
   {
      // Create a Model object to hold the geometry.
      Model model = new Model();

      // Give the Model a color.
      model.r = 0.0;
      model.g = 0.0;
      model.b = 1.0;

      // Add the geometry to the Model.
      Vertex v1 = new Vertex(3.0, 0.0, 0.0);
      Vertex v2 = new Vertex(0.0, 3.0, 0.0);
      Vertex v3 = new Vertex(0.0, 0.0, 3.0);

      LineSegment ls1 = new LineSegment(v1, v2);
      model.lineSegmentList.add( ls1 );
      LineSegment ls2 = new LineSegment(v2, v3);
      model.lineSegmentList.add( ls2 );
      LineSegment ls3 = new LineSegment(v3, v1);
      model.lineSegmentList.add( ls3 );

      // Create the Scene object that we shall render
      Scene scene = new Scene();

      // Add the Model to the Scene.
      scene.modelList.add(model);

      // Create a Camera object.
      Camera camera = new Camera();

      // Give the Camera a view matrix.
      camera.view2Identity();
/*
      // Set up the camera's view frustum.
      double fov    = 60.0;
      double aspect = 1.0;
      double near   = 1.0;
      double far    = 20.0;
      camera.projPerspective(fov, aspect, near, far);
*/
      // Set up the camera's view frustum.
      double left   = -1.0;
      double right  =  1.0;
      double top    =  1.0;
      double bottom = -1.0;
      double near   =  2.0;
      double far    =  20.0;
      camera.projFrustum(left, right, bottom, top, near, far);

      // Set up the camera's location and orientation.
      double eyex=0,    eyey=0,    eyez=6;
      double centerx=0, centery=0, centerz=0;
      double upx=0,     upy=1,     upz=0;
      camera.viewLookAt(eyex,    eyey,    eyez,
                        centerx, centery, centerz,
                        upx,     upy,     upz);

      // Add the Camera to the Scene.
      scene.setCamera(camera);

      //System.err.println( scene.toString() );

      // Create a framebuffer to render our scene into.
      int vp_width  = 512;
      int vp_height = 512;
      FrameBuffer fb = new FrameBuffer(vp_width, vp_height);
      // Give the framebuffer a nice background color.
      fb.clearFB(java.awt.Color.lightGray);
      // Render our scene into the frame buffer.
      SceneRender.renderScene(scene, fb);
      // Save the resulting image in a file.
      fb.dumpFB2File( String.format("PPM_Triangle_Frame%02d.ppm", 0) );

      // Spin the model 360 degrees.
      for (int i = 1; i <= 36; i++)
      {
         // Rotate the model by 10 degrees.
         scene.camera.viewRotate(10, 0, 1, 0);
         // Render again.
         fb.clearFB(java.awt.Color.lightGray);
         SceneRender.renderScene(scene, fb);
         fb.dumpFB2File(String.format("PPM_Triangle_Frame%02d.ppm", i));
      }
   }
}