/**
   A Camera data structure has a view matrix
   (which determines the scene's "camera" location)
   and a "projection matrix" (which determines the
   camera's view volume).
*/

public class Camera
{
   public Matrix viewMatrix;

   public double left;   // These numbers define the camera's "view volume".
   public double right;  // (In future renderers, these numbers will be
   public double bottom; //  encoded into a "projection matrix".)
   public double top;
   public double near;
   public double far;

   public boolean perspective = true;  // choose either perspective or parallel projection

   public Camera()
   {
      viewMatrix = new Matrix(1.0); // identity matrix
      left   = -1.0;
      right  =  1;
      bottom = -1.0;
      top    =  1;
      near   =  1.0;
      far    =  2.0;
   }


   //  http://www.opengl.org/sdk/docs/man2/xhtml/gluPerspective.xml
   public void projPerspective(double fovy, double aspect, double near, double far)
   {
      this.bottom = -near * Math.tan((Math.PI/180.0)*fovy/2.0);
      this.top    =  near * Math.tan((Math.PI/180.0)*fovy/2.0);
      this.left   = -top * aspect;
      this.right  =  top * aspect;
      this.near   = near;
      this.far    = far;  // we are not yet using the "far" parameter

      this.perspective = true;
   }


   //  http://www.opengl.org/sdk/docs/man2/xhtml/glFrustum.xml
   public void projFrustum(double left, double right, double bottom, double top, double near, double far)
   {
      this.left   = left;
      this.right  = right;
      this.bottom = bottom;
      this.top    = top;
      this.near   = near;
      this.far    = far;  // we are not yet using the "far" parameter

      this.perspective = true;
   }


   //  http://www.opengl.org/sdk/docs/man2/xhtml/glOrtho.xml
   public void projOrtho(double left, double right, double bottom, double top, double near, double far)
   {
      this.left   = left;
      this.right  = right;
      this.bottom = bottom;
      this.top    = top;
      this.near   = near;
      this.far    = far;  // we are not yet using the "far" parameter

      this.perspective = false;
   }


   //  http://www.opengl.org/sdk/docs/man2/xhtml/gluLookAt.xml
   public void viewLookAt(double eyex,    double eyey,    double eyez,
                          double centerx, double centery, double centerz,
                          double upx,     double upy,     double upz)
   {
      Vector F  = new Vector(centerx - eyex, centery - eyey, centerz - eyez);
      Vector UP = new Vector(upx, upy, upz);
      Vector f  = F.normalize();
      Vector up = UP.normalize();
      Vector s  = f.crossProduct(up);
      Vector u  = s.crossProduct(f);
      viewMatrix = viewMatrix.times(new Matrix(new Vector(s.x, u.x, -f.x, 0.0),
                                               new Vector(s.y, u.y, -f.y, 0.0),
                                               new Vector(s.z, u.z, -f.z, 0.0),
                                               new Vector(0.0, 0.0,  0.0, 1.0)));

      viewMatrix = viewMatrix.times(new Matrix(new Vector(-eyex, -eyey, -eyez)));
   }


   //  https://www.opengl.org/sdk/docs/man2/xhtml/glMatrixMode.xml
   //  https://www.opengl.org/sdk/docs/man2/xhtml/glLoadIdentity.xml
   public void view2Identity()
   {
      viewMatrix = new Matrix(1.0);
   }


   //  https://www.opengl.org/sdk/docs/man2/xhtml/glMatrixMode.xml
   //  https://www.opengl.org/sdk/docs/man2/xhtml/glRotate.xml
   public void viewRotate(double theta, double x, double y, double z)
   {
      viewMatrix = viewMatrix.times(new Matrix(theta, x, y, z));
   }


   //  https://www.opengl.org/sdk/docs/man2/xhtml/glMatrixMode.xml
   //  https://www.opengl.org/sdk/docs/man2/xhtml/glTranslate.xml
   public void viewTranslate(double x, double y, double z)
   {
      viewMatrix = viewMatrix.times(new Matrix( new Vector(x, y, z)));
   }
}