/**
   This FrameBuffer represents a two-dimensional array of pixel data
   stored as a one dimensional array in row-major order. The first
   row of pixels should be displayed as the top of the image (or
   window) that is made up of this pixel data.

   The "viewport" is a two-dimensional sub array of the frame buffer.
   The viewport is represented by its upper-left-hand corner and its
   lower-right-hand corner.

   Framebuffer and viewport coordinates act like Java Graphics
   coordinates; the positive x direction is to the right and
   the positive y direction is downward.

   This frame buffer also includes a z-buffer for representing depth
   information.
*/
import java.awt.Color;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileNotFoundException;
import java.io.IOException;

public class FrameBuffer
{
   public int width;          // the framebuffer's width
   public int height;         // the framebuffer's height
   public int[] pixel_buffer; // contains the pixel's color data for a rendered frame
   public double[] z_buffer;  // contains the pixel's depth data

   // Coordinates of the current viewport within the framebuffer.
   private int vp_ul_x; // upper-left-hand corner
   private int vp_ul_y;
   private int vp_lr_x; // lower-right-hand corner
   private int vp_lr_y;


   /**
      Construct a FrameBuffer with the given dimensions.
      Initialize the framebuffer to hold all black pixels.

      @param w Width of the FrameBuffer.
      @param h Height of the FrameBuffer.
   */
   public FrameBuffer(int w, int h)
   {
      this.width  = w;
      this.height = h;
      // Set the default viewport (to be the whole framebuffer).
      vp_ul_x = 0;
      vp_ul_y = 0;
      vp_lr_x = width - 1;
      vp_lr_y = height - 1;

      // Create the pixel buffer.
      pixel_buffer = new int[width * height * 3];

      // Create the z-buffer (the depth buffer).
      z_buffer = new double[width * height];

      // Initialize the pixel and z buffers.
      clearFB(Color.black);
   }


   /**
      Construct a FrameBuffer from a PPM image file.
      This can be used to initialize a FrameBuffer with a background image.

      @param inputFileName Must name a PPM image file with magic number P6.
   */
   public FrameBuffer(String inputFileName)
   {
      // Read the pixel data in a PPM file.
      // http://stackoverflow.com/questions/2693631/read-ppm-file-and-store-it-in-an-array-coded-with-c
      try
      {
         FileInputStream fis = new FileInputStream(inputFileName);

         // Read image format string "P6".
         String magicNumber = "";
         char c = (char)fis.read();
         while (c != '\n')
         {
            magicNumber += c;
            c = (char)fis.read();
         }
         if (! magicNumber.trim().startsWith("P6"))
         {
            System.err.printf("ERROR! Improper PPM number in file %s\n", inputFileName);
            System.exit(-1);
         }

         c = (char)fis.read();
         if ( '#' == c ) // read (and discard) IrfanView comment
         {
            while (c != '\n')
            {
               c = (char)fis.read();
            }
            c = (char)fis.read();
         }

         // Read image dimensions.
         String widthDim = "";
         while (c != ' ' && c != '\n')
         {
            widthDim += c;
            c = (char)fis.read();
         }
         this.width = Integer.parseInt(widthDim.trim());

         String heightDim = "";
         c = (char)fis.read();
         while (c != '\n')
         {
            heightDim += c;
            c = (char)fis.read();
         }
         this.height = Integer.parseInt(heightDim.trim());

         // Read image rgb dimensions (which we don't use).
         c = (char)fis.read();
         while (c != '\n')
         {
            c = (char)fis.read();
         }

         // Set the default viewport (to be the whole framebuffer).
         vp_ul_x = 0;
         vp_ul_y = 0;
         vp_lr_x = this.width - 1;
         vp_lr_y = this.height - 1;

         // Create the pixel buffer.
         pixel_buffer = new int[this.width * this.height * 3];

         // Create the z-buffer (the depth buffer).
         z_buffer = new double[this.width * this.height];

         // Initialize the pixel and z buffers.
         clearFB(Color.black);

         // Create a data array.
         byte[] pixelData = new byte[3];
         // Read pixel data, one pixel at a time, from the PPM file.
         for (int y = 0; y < this.height; y++)
         {
            for (int x = 0; x < this.width; x++)
            {
               if ( fis.read(pixelData, 0, 3) != 3 )
               {
                  System.err.printf("ERROR! Could not load %s\n", inputFileName);
                  System.exit(-1);
               }
               int r = pixelData[0];
               int g = pixelData[1];
               int b = pixelData[2];
               if (r < 0) r = 256+r;  // convert from signed byte to unsigned byte
               if (g < 0) g = 256+g;
               if (b < 0) b = 256+b;
               setPixelFB(x, y, new Color(r, g, b));
            }
         }
         fis.close();
      }
      catch (IOException e)
      {
         e.printStackTrace(System.err);
         System.err.printf("ERROR! Could not read %s\n", inputFileName);
         System.exit(-1);
      }
   }


   /**
      Get the width of the framebuffer.
   */
   public int getWidthFB()
   {
      return width;
   }


   /**
      Get the height of the framebuffer.
   */
   public int getHeightFB()
   {
      return height;
   }


   /**
      Clear the framebuffer (set a background color)
      and clear the framebuffer's z-buffer.
   */
   public void clearFB(Color c)
   {
      for (int y = 0; y < height; y++)
      {
         for (int x = 0; x < width; x++)
         {
            int index = (y*width + x)*3;
            pixel_buffer[index + 0] = c.getRed();
            pixel_buffer[index + 1] = c.getGreen();
            pixel_buffer[index + 2] = c.getBlue();
         }
      }
      // clear the z-buffer (the depth buffer)
      clearzFB();
   }


   /**
      Clear the framebuffer's z-buffer (the depth buffer).
   */
   public void clearzFB()
   {
      for (int y = 0; y < height; y++)
      {
         for (int x = 0; x < width; x++)
         {
            z_buffer[y*width + x] = 1.0;
         }
      }
   }


   /**
      Set the coordinates, within the FrameBuffer, of the
      viewport's upper-left-hand corner, width, and height.
      (Using upper-left-hand corner, width, and height is
       like Java's Rectangle class and Graphics.drawRect()
       method.)
   */
   public void setViewport(int vp_ul_x, int vp_ul_y, int width, int height)
   {
      this.vp_ul_x = vp_ul_x;
      this.vp_ul_y = vp_ul_y;
      this.vp_lr_x = vp_ul_x + width - 1;
      this.vp_lr_y = vp_ul_y + height - 1;
   }


   /**
      Get the width of the viewport.
   */
   public int getWidthVP()
   {
      return vp_lr_x - vp_ul_x + 1;
   }


   /**
      Get the height of the viewport.
   */
   public int getHeightVP()
   {
      return vp_lr_y - vp_ul_y + 1;
   }


   /**
      Clear the viewport (set a background color)
      and clear the viewport's z-buffer.
   */
   public void clearVP(Color c)
   {
      for (int y = vp_ul_y; y <= vp_lr_y; y++)
      {
         for (int x = vp_ul_x; x <= vp_lr_x; x++)
         {
            int index = (y*width + x)*3;
            pixel_buffer[index + 0] = c.getRed();
            pixel_buffer[index + 1] = c.getGreen();
            pixel_buffer[index + 2] = c.getBlue();
         }
      }
      // clear the z-buffer (the depth buffer)
      clearzVP();
   }


   /**
      Clear the viewport's z-buffer (the depth buffer).
   */
   public void clearzVP()
   {
      for (int y = vp_ul_y; y <= vp_lr_y; y++)
      {
         for (int x = vp_ul_x; x <= vp_lr_x; x++)
         {
            z_buffer[y*width + x] = 1.0;
         }
      }
   }


   /**
      Get the color of the pixel with coordinates
      (x,y) in the framebuffer.
   */
   public Color getPixelFB(int x, int y)
   {
      int index = (y*width + x)*3;
      int r = pixel_buffer[index + 0];
      int g = pixel_buffer[index + 1];
      int b = pixel_buffer[index + 2];
      return new Color(r, g, b);
   }


   /**
      Get the depth of the pixel with coordinates
      (x,y) in the framebuffer.
   */
   public double getDepthFB(int x, int y)
   {
      return z_buffer[y*width + x];
   }


   /**
      Set the color of the pixel with coordinates
      (x,y) in the framebuffer.
   */
   public void setPixelFB(int x, int y, Color c)
   {
      int index = (y*width + x)*3;
      pixel_buffer[index + 0] = c.getRed();
      pixel_buffer[index + 1] = c.getGreen();
      pixel_buffer[index + 2] = c.getBlue();
   }


   /**
      Set the depth of the pixel with coordinates
      (x,y) in the framebuffer.
   */
   public void setPixelFB(int x, int y, double z)
   {
      int index = (y*width + x)*3;
      z_buffer[y*width + x] = z;
   }


   /**
      Set the color and depth of the pixel with coordinates
      (x,y) in the framebuffer.
   */
   public void setPixelFB(int x, int y, Color c, double z)
   {
      int index = (y*width + x)*3;
      pixel_buffer[index + 0] = c.getRed();
      pixel_buffer[index + 1] = c.getGreen();
      pixel_buffer[index + 2] = c.getBlue();

      z_buffer[y*width + x] = z;
   }


   /**
      Get the color of the pixel with coordinates
      (x,y) relative to the current viewport.
   */
   public Color getPixelVP(int x, int y)
   {
      int index = ((vp_ul_y + y)*width + vp_ul_x + x)*3;
      int r = pixel_buffer[index + 0];
      int g = pixel_buffer[index + 1];
      int b = pixel_buffer[index + 2];
      return new Color(r, g, b);
   }


   /**
      Get the depth of the pixel with coordinates
      (x,y) relative to the current viewport.
   */
   public double getDepthVP(int x, int y)
   {
      return z_buffer[(vp_ul_y + y)*width + vp_ul_x + x];
   }


   /**
      Set the color of the pixel with coordinates
      (x,y) relative to the current viewport.
   */
   public void setPixelVP(int x, int y, Color c)
   {
      int index = ((vp_ul_y + y)*width + vp_ul_x + x)*3;
      pixel_buffer[index + 0] = c.getRed();
      pixel_buffer[index + 1] = c.getGreen();
      pixel_buffer[index + 2] = c.getBlue();
   }


   /**
      Set the depth of the pixel with coordinates
      (x,y) relative to the current viewport.
   */
   public void setPixelVP(int x, int y, double z)
   {
      int index = ((vp_ul_y + y)*width + vp_ul_x + x)*3;
      z_buffer[(vp_ul_y + y)*width + vp_ul_x + x] = z;
   }


   /**
      Set the color and depth of the pixel with coordinates
      (x,y) relative to the current viewport.
   */
   public void setPixelVP(int x, int y, Color c, double z)
   {
      int index = ((vp_ul_y + y)*width + vp_ul_x + x)*3;
      pixel_buffer[index + 0] = c.getRed();
      pixel_buffer[index + 1] = c.getGreen();
      pixel_buffer[index + 2] = c.getBlue();

      z_buffer[(vp_ul_y + y)*width + vp_ul_x + x] = z;
   }


   /**
      Convert the z-buffer into a framebuffer that holds
      a gray scale color value representing each pixels
      "depth".

      z-buffer values are between -1.0 and 1.0.
      -1.0 represents the front of the scene.
       1.0 represents the back of the scene.
      Make things near the front look darker and
      things near the back look lighter.
   */
   public FrameBuffer convertZB2FB()
   {
      FrameBuffer zfb = new FrameBuffer(this.width, this.height);
      zfb.setViewport(this.vp_ul_x, this.vp_ul_y, this.vp_lr_x, this.vp_lr_y);

      // Copy the current z-buffer into the new framebuffer's pixel buffer.
      // Copy each entry of the current z-buffer into each of the the
      // corresponding pixel's r, g, and b values.
      for (int i = 0; i < this.z_buffer.length; i++)
      {
         // this.z_buffer[i] is between -1 and +1 (front to back)
         // gray is between 0 and 255 (dark to light)
         int gray = (int)( 127.5 + (127.5 * this.z_buffer[i]) );
         zfb.pixel_buffer[3*i + 0] = gray;
         zfb.pixel_buffer[3*i + 1] = gray;
         zfb.pixel_buffer[3*i + 2] = gray;
      }

      // Copy the current z-buffer into the new framebuffer's z-buffer.
      for (int i = 0; i < this.z_buffer.length; i++)
      {
         zfb.z_buffer[i] = this.z_buffer[i];
      }

      return zfb;
   }


   /**
      Convert the viewport into a framebuffer.
   */
   public FrameBuffer convertVP2FB()
   {
      FrameBuffer vp_fb = new FrameBuffer(this.getWidthVP(), this.getHeightVP());

      // Copy the current viewport into the new framebuffer's pixel buffer.
      for (int y = 0; y < this.getHeightVP(); y++)
         for (int x = 0; x < this.getWidthVP(); x++)
         {
            vp_fb.setPixelVP( x, y, this.getPixelVP(x, y), this.getDepthVP(x, y) );
         }

      return vp_fb;
   }


   /**
      Write the framebuffer to the specified file.
   */
   public void dumpFB2File(String filename)
   {
      dumpPixels2File(0, 0, width-1, height-1, filename);
   }


   /**
      Write the viewport to the specified file.
   */
   public void dumpVP2File(String filename)
   {
      dumpPixels2File(vp_ul_x, vp_ul_y, vp_lr_x, vp_lr_y, filename);
   }


   /**
      Write a rectangular sub array of pixels to the specified file.

      http://stackoverflow.com/questions/2693631/read-ppm-file-and-store-it-in-an-array-coded-with-c
   */
   public void dumpPixels2File(int ul_x, int ul_y, int lr_x, int lr_y, String filename)
   {
      int p_width  = lr_x - ul_x + 1;
      int p_height = lr_y - ul_y + 1;

      FileOutputStream fos = null;
      try  // open the file
      {
         fos = new FileOutputStream(filename);
      }
      catch (FileNotFoundException e)
      {
         e.printStackTrace(System.err);
         System.err.printf("ERROR! Could not open file %s\n", filename);
         System.exit(-1);
      }
      //System.err.printf("Created file %s\n", filename);

      try  // write data to the file
      {
         // write the PPM header information first
         fos.write( ("P6\n" + p_width + " " + p_height + "\n" + 255 + "\n").getBytes() );

         // write the pixel data to the file
         byte[] temp = new byte[p_width*3];  // array to hold one row of data
         for (int n = 0; n < p_height; n++)
         {  // write one row of pixels at a time,

            // read from the top row of the data buffer
            // down towards the bottom row
            for (int i = 0; i < temp.length; i++)
               temp[i] = (byte)pixel_buffer[((ul_y+n)*width + ul_x)*3 + i];
            /*
            // read from the bottom row of the data buffer
            // up towards the top row
            for (int i = 0; i < temp.length; i++)
               temp[i] = (byte)pixel_buffer[((lr_y-n)*width + ul_x)*3 + i];
            */
            fos.write(temp); // write one row of data
         }
      }
      catch (IOException e)
      {
         e.printStackTrace(System.err);
         System.err.printf("ERROR! Could not write to file %s\n", filename);
         System.exit(-1);
      }

      try
      {
         fos.close();
      }
      catch (IOException e)
      {
         e.printStackTrace(System.err);
         System.err.printf("ERROR! Could not close file %s\n", filename);
         System.exit(-1);
      }
   }//dumpPixels2File()


   /**
      A simple test of the framebuffer.
      It fills the framebuffer with a test pattern.
   */
   public void fbTest()
   {
      int w = this.width;

      for (int y = 0; y < this.height; y++)
      {
         for (int x = 0; x < this.width; x++)
         {
            int gray = (x|y)%255;
            this.pixel_buffer[(y*w + x)*3 + 0] = gray;
            this.pixel_buffer[(y*w + x)*3 + 1] = gray;
            this.pixel_buffer[(y*w + x)*3 + 2] = gray;
         }
      }
   }//fbTest()()


   /**
      A main() method for testing the FrameBuffer class.
   */
   public static void main(String[] args)
   {
      int w = 512;
      int h = 512;
      FrameBuffer fb = new FrameBuffer(w, h);
      fb.fbTest();  // fill the framebuffer with a test pattern
      fb.dumpFB2File("test0.ppm");
      fb.setViewport(64, 64, 64+127, 64+255);  // 128 by 256
      fb.clearVP( Color.red );
      fb.dumpFB2File("test1.ppm");
      fb.dumpVP2File("test2.ppm");
      fb.dumpPixels2File(32, 256-64, 511-64, 255+64, "test3.ppm"); // 416 by 128

      for (int i = 0; i < 256+127; i++)
         fb.setPixelFB(i, 128, Color.blue, 0);
      fb.dumpFB2File("test4.ppm");

      for (int i = 0; i < 255; i++)
         fb.setPixelVP(64, i, Color.blue, 0);
      fb.dumpFB2File("test5.ppm");
      fb.dumpVP2File("test6.ppm");
   }//main()
}