/**
   A Matrix object has four Vector objects.

   The four Vector objects represent the four column vectors of a 4-by-4 matrix (as in a Linear Algebra course).

   A matrix represents a "transformation" of 3-dimensional space (a translation, rotation, or scaling transformation).

   In computer graphics, the points of 3-dimensional space are represented using 4-dimensional homogeneous coordinates, so each transformation of 3-dimensional space is represented by a 4-by4 matrix.
*/

public class Matrix
{
   public Vector v1, v2, v3, v4;  // these are column vectors


   public Matrix()
   {
      v1 = new Vector();
      v2 = new Vector();
      v3 = new Vector();
      v4 = new Vector();
   }


   public Matrix(Vector v)   // translation matrix
   {
      v1 = new Vector( 1.0, 0.0, 0.0, 0.0 );
      v2 = new Vector( 0.0, 1.0, 0.0, 0.0 );
      v3 = new Vector( 0.0, 0.0, 1.0, 0.0 );
      v4 = new Vector( v.x, v.y, v.z, 1.0 );
   }


   public Matrix(double d)  // diagonal matrix
   {
      v1 = new Vector(   d, 0.0, 0.0, 0.0 );
      v2 = new Vector( 0.0,   d, 0.0, 0.0 );
      v3 = new Vector( 0.0, 0.0,   d, 0.0 );
      v4 = new Vector( 0.0, 0.0, 0.0, 1.0 );
   }


   public Matrix(double x, double y, double z)  // scaling matrix
   {
      v1 = new Vector(   x, 0.0, 0.0, 0.0 );
      v2 = new Vector( 0.0,   y, 0.0, 0.0 );
      v3 = new Vector( 0.0, 0.0,   z, 0.0 );
      v4 = new Vector( 0.0, 0.0, 0.0, 1.0 );
   }


   public Matrix(double theta, double x, double y, double z) // rotation matrix
   {
      //  http://www.opengl.org/sdk/docs/man2/xhtml/glRotate.xml
      double norm = Math.sqrt(x*x + y*y + z*z);
      double ux = x/norm;
      double uy = y/norm;
      double uz = z/norm;

      double c = Math.cos( (Math.PI/180.0)*theta );
      double s = Math.sin( (Math.PI/180.0)*theta );

      v1 = new Vector( ux*ux*(1-c)+c,      uy*ux*(1-c)+(uz*s), uz*ux*(1-c)-(uy*s), 0.0 );
      v2 = new Vector( ux*uy*(1-c)-(uz*s), uy*uy*(1-c)+c,      uz*uy*(1-c)+(ux*s), 0.0 );
      v3 = new Vector( ux*uz*(1-c)+(uy*s), uy*uz*(1-c)-(ux*s), uz*uz*(1-c)+c,      0.0 );
      v4 = new Vector( 0.0,                0.0,                0.0,                1.0 );
   }


   public Matrix(Vector _v1, Vector _v2, Vector _v3, Vector _v4)
   {
      v1 = _v1;
      v2 = _v2;
      v3 = _v3;
      v4 = _v4;
   }


   public void setColumn(int col, double x, double y, double z, double w)
   {
      if (col == 1)
      {
         v1.set(x, y, z, w);
      }
      else if (col == 2)
      {
         v2.set(x, y, z, w);
      }
      else if (col == 3)
      {
         v3.set(x, y, z, w);
      }
      else if (col == 4)
      {
         v4.set(x, y, z, w);
      }
   }


   public Matrix times(double s)  // scalar times Matrix
   {
      return new Matrix( v1.times(s), v2.times(s), v3.times(s), v4.times(s));
   }


   public Matrix times(Matrix m)  // Matrix times Matrix
   {
      return new Matrix( times(m.v1), times(m.v2), times(m.v3), times(m.v4) );
   }


   public Vector times(Vector v)  // Matrix times Vector
   {
      return v1.times(v.x).plus(v2.times(v.y).plus(v3.times(v.z).plus(v4.times(v.w))));
   /*
      Vector v1x = v1.times(v.x);
      Vector v2y = v2.times(v.y);
      Vector v3z = v3.times(v.z);
      Vector v4w = v4.times(v.w);
      Vector sum1 = v1x.plus(v2y);
      Vector sum2 = sum1.plus(v3z);
      Vector sum3 = sum2.plus(v4w);
      return sum3;
   */
   }


   public Vertex times(Vertex v)  // Matrix times Vertex
   {
      Vector sum = v1.times(v.x).plus(v2.times(v.y).plus(v3.times(v.z).plus(v4.times(v.w))));
      Vertex temp = new Vertex(sum.x, sum.y, sum.z, sum.w);
      return temp;
   }


   public String toString()
   {
      String result = "";
      int p = 5;      // the precision for the following format string
      int w = p + 4;  // the width for the folowing format string
      String format = "% "+w+"."+p+"f  % "+w+"."+p+"f  % "+w+"."+p+"f  % "+w+"."+p+"f";
      result += String.format("[[" + format + "]\n",  v1.x, v2.x, v3.x, v4.x);
      result += String.format(" [" + format + "]\n",  v1.y, v2.y, v3.y, v4.y);
      result += String.format(" [" + format + "]\n",  v1.z, v2.z, v3.z, v4.z);
      result += String.format(" [" + format + "]]\n", v1.w, v2.w, v3.w, v4.w);
    //result += String.format("[[% .5f  % .5f  % .5f  % .5f]\n",  v1.x, v2.x, v3.x, v4.x);
    //result += String.format(" [% .5f  % .5f  % .5f  % .5f]\n",  v1.y, v2.y, v3.y, v4.y);
    //result += String.format(" [% .5f  % .5f  % .5f  % .5f]\n",  v1.z, v2.z, v3.z, v4.z);
    //result += String.format(" [% .5f  % .5f  % .5f  % .5f]]\n", v1.w, v2.w, v3.w, v4.w);
      return result;
   }
}