/**
   A Model data structure represents a distinct geometric object (in a Scene). A Model data structure is mainly a list of LineSegment objects. The line segments represent the geometric object as a "wire-frame", that is, the geometric object is drawn as a collection of "edges". This is a fairly simplistic way of doing 3D graphics and we will improve this in a later renderer.

   See
      http://en.wikipedia.org/wiki/Wire-frame_model
   or
      https://www.google.com/search?q=graphics+wireframe&tbm=isch
*/
import java.util.List;
import java.util.LinkedList;
import java.awt.Color;

public class Model
{
   public List<LineSegment> lineSegmentList = new LinkedList<LineSegment>();

   public double r, g, b;  // color of this model


   public Model()
   {
      setColor(0.0, 0.0, 0.0);  // default to black
   }


   /**
      Duplicate a Model object.
   */
   public Model(Model model1)
   {
      // Copy everything except the list of line segments.
      this.r = model1.r;
      this.g = model1.g;
      this.b = model1.b;
   }//duplicating constructor


   /**
      Add a LineSegment to this model's list of line segments.
   */
   public void addLineSegment(LineSegment ls)
   {
      this.lineSegmentList.add(ls);
   }


   /**
      Set the color of this model to (r,g,b).
   */
   public void setColor(double r, double g, double b)
   {
      this.r = r;
      this.g = g;
      this.b = b;

      if ( (r < 0.0) || (r > 1.0)
        || (g < 0.0) || (g > 1.0)
        || (b < 0.0) || (b > 1.0) )
      {
         System.err.println("ERROR! Invalid double color for model");
         System.err.println( this.toString() );
         System.exit(-1);
      }
   }


   public void setColor(int r, int g, int b)
   {
      this.r = ((float)r)/(float)255;
      this.g = ((float)g)/(float)255;
      this.b = ((float)b)/(float)255;

      if ( (r < 0) || (r > 255)
        || (g < 0) || (g > 255)
        || (b < 0) || (b > 255))
      {
         System.err.println("ERROR! Invalid int color for model");
         System.err.println( this.toString() );
         System.exit(-1);
      }
   }


   public void setColor(Color c)
   {
      setColor(c.getRed(), c.getGreen(), c.getBlue());
   }


   public Color getColor()
   {
      return new Color((float)this.r, (float)this.g, (float)this.b);
   }


   /**
      For debugging.
   */
   public String toString()
   {
      String result = "";
      result += "This Model has " + lineSegmentList.size() + " line segments\n";
      result += String.format("<r,g,b> = <% .5f  % .5f  % .5f>\n", r, g, b);
      //result = "Printing out this Model's " + lineSegmentList.size() + " Line segments:\n";
      for (LineSegment ls : this.lineSegmentList)
      {
         result += ls.toString();
      }
      //result += "Done printing out Model\n";
      return result;
   }
}