/**
   A Vertex object has four doubles which represent the (homogeneous) coordinates of a point in 3-dimensional space. The fourth, homogeneous, coordinate will usually be 1, but in some stages of the graphics rendering pipeline it will be some other (non-zero) number.

   When a Vertex object is created in a client program, before the Vertex object moves down the graphics  rendering pipeline, the coordinates in the Vertex will be in "world coordinates".

   As a Vertex object moves down the graphics rendering pipeline, the coordinates in the Vertex will be transformed from one coordinate system to another (from world to view to clip to viewport coordinates).
*/

public class Vertex
{
   public double x, y, z, w;  // a vertex in homogenous coordinates


   public Vertex()
   {
      this(0.0, 0.0, 0.0, 1.0);
   }


   public Vertex(double x, double y, double z)
   {
      this(x, y, z, 1.0);
   }


   public Vertex(float[] v)
   {
      this(v[0], v[1], v[2], v[3]);
   }


   public Vertex(double x, double y, double z, double w)
   {
      set(x, y, z, w);
   }


   public Vertex(Vertex v) // a "copy constructor"
   {
      set(v.x, v.y, v.z, v.w);
   }


   private void set(double x, double y, double z)
   {
      set(x, y, z, 1.0);
   }


   private void set(double x, double y, double z, double w)
   {
      this.x = x;
      this.y = y;
      this.z = z;
      this.w = w;
   }


   /**
      Vertex plus Vector returns a Vertex.
      This can be used to translate a Vertex (but a translation transformation would be better).
   */
   public Vertex plus(Vector v)
   {
      return new Vertex( this.x+(v.x), this.y+(v.y), this.z+(v.z), this.w+(v.w) );
   }


   /**
      Vertex minus Vertex returns a Vector.
      Used to find the displacement vector between two vertices.
   */
   public Vector minus(Vertex v)
   {
      return new Vector( this.x-(v.x), this.y-(v.y), this.z-(v.z), this.w-(v.w) );
   }


   /**
      Scalar times Vertex returns a Vertex.
      Needed for the perspective division step in the rendering pipeline.
   */
   public Vertex times(double scalar)
   {
      return new Vertex(scalar*this.x, scalar*this.y, scalar*this.z, scalar*this.w);
   }


   /**
      For debugging.
   */
   public String toString()
   {
      // Here is one way to get programable precision and width.
      int p = 5;     // the precision for the following format string
      int t = p + 4; // the width for the folowing format string
      String format =      "(x,y,z,w) = (% "+t+"."+p+"f  % "+t+"."+p+"f  % "+t+"."+p+"f  % "+t+"."+p+"f)\n";
      return String.format( format, x, y, z, w);
    //return String.format("(x,y,z,w) = (% .5f  % .5f  % .5f  % .5f)", x, y, z, w);
   }
}