/**
   Project line segments from view coordinates to clip coordinates.

   For each Model in a Scene, walk the Model's list of LineSegment objects.
   For each Vertex object in each LineSgement object, use the Scene's
   "projection matrix" to project the Vertex object into the front plane
   of the Scene's view volume.

   After projecting the vertex into the front plane of the view volume,
   "normalize" the coordinates of the vertex so that the front *face* of
   the Scene's view volume is represented by points with x and y coordinates
   between -1 and 1. (These new coordinates are called "clip coordinates" or
   "normalized device coordinates".)
*/

public class P2_View2Clip
{
   /**
      For each line segment in the model, project each
      vertex to the front plane of the view volume and
      then transform each vertex from view coordinates
      to clip coordinates.
   */
   public static void p2_View2Clip(Model model, Scene scene)
   {
      double left   = scene.camera.left;   // These numbers define the camera's "view volume".
      double right  = scene.camera.right;  // (In future renderers, these numbers will be
      double bottom = scene.camera.bottom; //  encoded into a "projection matrix".)
      double top    = scene.camera.top;
      double near   = scene.camera.near;
      double far    = scene.camera.far;

      for (LineSegment ls : model.lineSegmentList)
      {
         // For each vertex in the line segment, compute its projected (x,y) coordinates.
         double v0_xp = -near * (ls.v[0].x / ls.v[0].z);  // xp = near * (xv / -zv)
         double v0_yp = -near * (ls.v[0].y / ls.v[0].z);  // yp = near * (yv / -zv)
         double v1_xp = -near * (ls.v[1].x / ls.v[1].z);
         double v1_yp = -near * (ls.v[1].y / ls.v[1].z);

         if ( ! scene.camera.perspective ) // do parallel projection instead
         {
            v0_xp = ls.v[0].x;
            v0_yp = ls.v[0].y;
            v1_xp = ls.v[1].x;
            v1_yp = ls.v[1].y;
         }

         // For each vertex, normalize the projected coordinates to "clip coordinates".
         double v0_xc = ((2 * v0_xp) - right - left)/(right - left);
         double v0_yc = ((2 * v0_yp) - top - bottom)/(top - bottom);
         double v1_xc = ((2 * v1_xp) - right - left)/(right - left);
         double v1_yc = ((2 * v1_yp) - top - bottom)/(top - bottom);


         // Construct new Vertex objects containing the (homogeneous) clip coordinates.
         ls.v[0] = new Vertex( v0_xc, v0_yc, near, 1 );
         ls.v[1] = new Vertex( v1_xc, v1_yc, near, 1 );
      }
   }
}