/**
   Clip any line segment that sticks out of the front face of the Scene's view volume.

   For each Model in a Scene, walk the Model's list of LineSegment objects.
   For each LineSegment object, clip the line segment so that both of its
   vertices are within the standard (normalized) clipping rectangle,
      -1 <= x <= +1  and
      -1 <= y <= +1.

   Here is an outline of the clipping algorithm (this algorithm is a
   simplification of the Liang-Barsky Parametric Line Clipping algorithm).

   First we create an empty line segment list called newLineSegmentList.
   This list will hold the line segments that have been completely clipped,
   that is, both of their endpoint are within the standard clipping rectangle.
   Then we loop through the model's line segment list and process each line
   segment, using the following steps.
     1) Remove a line segment from the model's line segment list.
     2) Test if the line segment no longer needs to be clipped, i.e., both of
        its vertices are within the normalized clipping rectangle. If this is
        the case, insert this line segment into newLineSegmentList and go on to
        the next line segment.
     2) Test if the line segment should be "trivially rejected". A line segment
        is "trivially rejected" if it is on the wrong side of any of the four
        lines that bound the normalized clipping region (i.e., the four lines x=1,
        x=-1, y=1, y=-1). If so, then this line segment is neither put back into
        the model's line segment list, nor is it put into newLineSegmentList. So
        the line segment gets deleted from further processing.

        Notice that a line like the following one is trivially deleted because it
        is on the "wrong" side of the line x=1.

                           x=1
                            |                      v1
                            |                    /
                 +----------+                  /
                 |          |                /
                 |          |              /
                 |          |            /
                 |          |          /
                 |          |        /
                 +----------+      /
                            |    /
                            |  v0

        But the following line is NOT trivially deleted because, even though it
        is completely outside of the normalized clipping region, this line is not
        entirely on the wrong side of any one of the four lines x=1, x=-1, y=1,
        or y=-1. The line below will get clipped at least one time (either on the
        line x=1 or the line y=-1) before it is a candidate for "trivial deletion".
        Notice that the line below could even be clipped twice, first on y=1,
        then on x=1, before it can be trivially rejected (by being on the wrong
        side of y=-1).
                           x=1
                            |                      v1
                            |                    /
                 +----------+                  /
                 |          |                /
                 |          |              /
                 |          |            /
                 |          |          /
                 |          |        /
                 +----------+      /
                            |    /
                            |  /
                             /
                           /
                        v0  |
                            |

     3) If the line segment has been neither accepted nor rejected, then it must need
        to be clipped. So we test the line segment against each of the four clipping
        lines, x=1, x=-1, y=1, and y=-1, to determine if the line segment crosses one
        of those lines. We clip the line segment against the first line which we find
        that it crosses. The we put the resulting clipped line segment back in the
        model's line segment list and continue the loop with another line segment.
        (Notice that we only clip against the first clipping line which the segment
        is found to cross. We do not continue to test against the other clipping lines.
        This is because it may be the case, after just one clip, that the line segment
        is now a candidate for trivial accept or reject. So rather than test the line
        segment against several more clipping lines (which may be useless tests) it is
        more efficient to put the (clipped) line segment back into the model's line
        segment list and then wait for the line segment to return to the front of the
        list. Then we can check it for trivial accept or reject.)

        When we clip a line segment against a clipping line, it is always the case that
        one endpoint of the line segment is on the "right" side of the clipping line and
        the other endpoint of the line segment is on the "wrong" side of the clipping line.
        In the following picture, assume that v0 is on the "right" side of the clipping
        line (x=1) and v1 is on the "wrong" side. So v1 needs to be clipped off the line
        segment and replaced by a new endpoint.

                             x=1
                              |
                        v1    |
                          \   |
                            \ |
                              \
                              | \
                              |   \
                              |     v0
        Represent points p(t) on the line segment between v0 and v1 with the following
        parametric equation.
                  p(t) = (1-t) * v0 + t * v1  with  0 <= t <= 1
        Notice that this equation parameterizes the line segment starting with v0
        at t=0 (on the "wrong side") and ending with v1 at t=1 (on the "right side").
        We need to find the value of t when the line segment crosses the clipping
        line x=1. Let v0=(x0, y0) and let v1=(x1, y1). Then the above parametric
        equation becomes the two component equations
                 x(t) = (1-t) * x0 + t * x1,
                 y(t) = (1-t) * y0 + t * y1  with  0 <= t <= 1.
        Since the clipping line in this example is x=1, we need to solve the equation
        x(t) = 1 for t. So we need to solve
                  (1-t) * x0 + t * x1 = 1
        for t. Here are a few algebra steps.
                  x0 - t * x0 + t * x1 = 1
                  x0 + t * (x1 - x0) = 1
                       t * (x1 - x0) = 1 - x0
                       t = (1 - x0)/(x1 - x0)
        We get similar equations for t if we clip against the other clipping lines
        (x=-1, y=1, or y=-1) and we assume that v0 is on the "wrong side" and v1 is
        on the "right side". With the above value for t, the new point p(t) that
        replaces v0 can easily be computed.
                             x=1
                              |
                        v1    |
                          \   |
                            \ |
                              v0=p( (1 - x0)/(x1 - x0) )
                              |
                              |
                              |
         Finally, the new line segment between v1 and the new v0 is placed in the
         model's line segment list so that it can be checked once again to see if
         it can be trivially accepted, trivially rejected, or clipped again.
*/
import java.util.List;
import java.util.LinkedList;

public class P3_Clipping
{
   /**
      Walk the list of LineSegment objects and if any line segment
      sticks out of the normalized view rectangle, then clip the
      line segment so that it is contained in the normalized view
      rectangle.
   */
   public static void p3_Clipping(Model model)
   {
      // Create a new, empty LineSegment list to hold the clipped line segments.
      List<LineSegment> newLineSegmentList = new LinkedList<LineSegment>();

      while ( ! model.lineSegmentList.isEmpty() )
      {
         LineSegment ls = model.lineSegmentList.remove(0);
         // make local copies of several values
         Vertex v0 = ls.v[0];
         Vertex v1 = ls.v[1];
         double x0 = v0.x;
         double y0 = v0.y;
         double z0 = v0.z;
         double x1 = v1.x;
         double y1 = v1.y;
         double z1 = v1.z;

         if ( ! ( Math.abs( x0 ) > 1
               || Math.abs( y0 ) > 1
               || Math.abs( x1 ) > 1
               || Math.abs( y1 ) > 1 ) )
         {  // This line segment doesn't need to be clipped, so put it in the new list.
            newLineSegmentList.add(ls);
         }
         else  // need to either delete or clip this line segment
         {
            // first check for trivial delete
            if ( (x0 >  1 && x1 >  1)   // to the right of the line x = 1
              || (x0 < -1 && x1 < -1)   // to the left of the line x = -1
              || (y0 >  1 && y1 >  1)   // above the line y = 1
              || (y0 < -1 && y1 < -1) ) // below the line y = -1
            {/* trivial delete; the line segment will be deleted after this if-statement */
               //System.err.println("Trivial delete.");
            }/* not trivial delete; the line segment must get clipped */
            else if (x0 > 1 || x1 > 1)  // ls crosses the line x = 1
            {
               // clip to the x = 1 line
               //System.err.println("clip to x = 1");
               if (x1 > 1)
               {  // create a new LineSegment containing previous v[0] and new v[1]
                  interpolateNewLineSegment( v0, v1, true, model.lineSegmentList, 1 );
               }
               else if (x0 > 1)
               {  // create a new LineSegment containing previous v[1] and new v[0]
                  interpolateNewLineSegment( v1, v0, false, model.lineSegmentList, 1 );
               }
               else  // should never get here
               {
                  System.err.println("Clipping to x = 1 dropped a line segment!");
               }
            }
            else if (x0 < -1 || x1 < -1)  // ls crosses the line x = -1
            {
               // clip to the x = -1 line
               //System.err.println("clip to x = -1");
               if (x1 < -1)
               {  // create a new LineSegment containing previous v[0] and new v[1]
                  interpolateNewLineSegment( v0, v1, true, model.lineSegmentList, 2 );
               }
               else if (x0 < -1)
               {  // create a new LineSegment containing previous v[1] and new v[0]
                  interpolateNewLineSegment( v1, v0, false, model.lineSegmentList, 2 );
               }
               else  // should never get here
               {
                  System.err.println("Clipping to x = -1 dropped a line segment!");
               }
            }
            else if (y0 > 1 || y1 > 1)  // ls crosses the line y = 1
            {
               // clip to the y = 1 line
               //System.err.println("clip to y = 1");
               if (y1 > 1)
               {  // create a new LineSegment containing previous v[0] and new v[1]
                  interpolateNewLineSegment( v0, v1, true, model.lineSegmentList, 3 );
               }
               else if (y0 > 1)
               {  // create a new LineSegment containing previous v[1] and new v[0]
                  interpolateNewLineSegment( v1, v0, false, model.lineSegmentList, 3 );
               }
               else  // should never get here
               {
                  System.err.println("Clipping to y = 1 dropped a line segment!");
               }
            }
            else if (y0 < -1 || y1 < -1)  // ls crosses the line y = -1
            {
               // clip to the y = -1 line
               //System.err.println("clip to y = -1");
               if (y1 < -1)
               {  // create a new LineSegment containing previous v[0] and new v[1]
                  interpolateNewLineSegment( v0, v1, true, model.lineSegmentList, 4 );
               }
               else if (y0 < -1)
               {  // create a new LineSegment containing previous v[1] and new v[0]
                  interpolateNewLineSegment( v1, v0, false, model.lineSegmentList, 4 );
               }
               else  // should never get here
               {
                  System.err.println("Clipping to y = -1 dropped a line segment!");
               }
            }
            else
            {// should never get here
               System.err.println("Clipping error!");
               System.exit(-1);
            }
         }
      }// continue with the next line segment

      // Swap the new line segment list for the old list.
      model.lineSegmentList =  newLineSegmentList;
   }


   /**
      This method takes in two vertices, one that is on the "right" side
      of a clipping line and the other that is on the "wrong" side of the
      clipping line, a boolean which specifies which of the first two
      vertices is v0 (true means v_inside is v0), the linked list into
      which the new clipped line segment should be put, and an integer
      which specifies which clipping line to use, where
         eqn_number == 1  means the clipping line x = 1
         eqn_number == 2  means the clipping line x =-1
         eqn_number == 3  means the clipping line y = 1
         eqn_number == 4  means the clipping line y =-1

      When this method returns, it will have placed in lineSegmentList
      a line segment from v_inside to the new interpolated vertex.
   */
   public static void interpolateNewLineSegment(Vertex v_inside,
                                                Vertex v_outside,
                                                boolean direction, // true means from inside to outside
                                                List<LineSegment> lineSegmentList,
                                                int eqn_number)
   {
      // interpolate a new vertex between v_outside and v_inside
      Vertex v_new = interpolateNewVertex(v_inside, v_outside, eqn_number);

      // create a new LineSegment (with the correct orientation)
      LineSegment ls_new;
      if (direction)
         ls_new = new LineSegment( v_inside, v_new );
      else
         ls_new = new LineSegment( v_new, v_inside );

      // add the new LineSegment to the end of the original linked list
      // since the line segment may need to be clipped again
      lineSegmentList.add(ls_new);
   }


   /**
      This method takes in two vertices, one that is on the "right" side
      of a clipping line and the other that is on the "wrong" side of the
      clipping line, and an integer which specifies which clipping line
      to use, where
         eqn_number == 1  means the clipping line x = 1
         eqn_number == 2  means the clipping line x =-1
         eqn_number == 3  means the clipping line y = 1
         eqn_number == 4  means the clipping line y =-1

      This method returns the vertex that is the intersection point
      between the given line segment and the given clipping line.

      This method solves for the value of t for which the parametric
      equation
                  p(t) = (1-t) * v_outside + t * v_inside
      intersects the given clipping line (notice that the equation
      is parameterized so that we move from the outside vertex towards
      the inside vertex as t increases from 0 to 1). The solved for
      value of t is then plugged into the parametric formula to get
      the coordinates of the intersection point.
   */
   private static Vertex interpolateNewVertex(Vertex v_inside,
                                              Vertex v_outside,
                                              int eqn_number)
   {
      //System.err.println("Create new vertex.");
      // make local copies of several values
      double vix =  v_inside.x; // "i" for "inside"
      double viy =  v_inside.y;
      double vox = v_outside.x; // "o" for "outside"
      double voy = v_outside.y;
      // interpolate between v_outside and v_inside
      double t = 0.0;
      if (1 == eqn_number)            // clip to x = 1
         t = (1 - vox) / (vix - vox);
      else if (2 == eqn_number)       // clip to x = -1
         t = (-1 - vox) / (vix - vox);
      else if (3 == eqn_number)       // clip to y = 1
         t = (1 - voy) / (viy - voy);
      else if (4 == eqn_number)       // clip to y = -1
         t = (-1 - voy) / (viy - voy);

      t = t + .000001;  /*** keep the new vertex off the edge!! ***/

      Vertex v_new = new Vertex();

      // use the value of t to interpolate the coordinates of the new vertex
      v_new.x = (1-t) * vox + t * vix;
      v_new.y = (1-t) * voy + t * viy;
    //v_new.z = 1.0;

   //System.err.printf("P3 [% .6f % .6f]\n", vox, voy);
   //System.err.printf("P3 t = % .6f\n", t);
   //System.err.printf("P3 <% .6f % .6f>\n", v_new.x,  v_new.y);

      return v_new;
   }

}