/**
   Convert line segments from normalized device coordinates to viewport coordinates.

   For each Model in a Scene, walk the Model's list of LineSegment objects.
   For each Vertex object in each LineSgement object, transform the coordinates
   of the vertex from "Normalized Device Coordinates" to "pixel coordinates" in
   the viewport.

   This is the last transformation before we rasterize the line segments.
   This is also the first stage of the pipeline that needs to know about the framebuffer.

   The "normalized device coordinates" are all between -1 and +1, so
        -1 <= x_ndc <= +1
        -1 <= y_ndc <= +1.
   The horizontal viewport coordinate should be between 0 and vpWidth.
   The vertical viewport coordinate should be between 0 and vpHeight.
   So
         0 <= x_vp <= vpWidth
         0 <= y_vp <= vpHeight.
   Notice that viewport coordinates are still floating point numbers (they
   are not yet integer coordinates; that's done in the rasterizer).

   The upper left hand corner of normalized device coordinates, (-1, 1), should
   map to the upper left hand corner of the viewport, (0, 0). And the lower right
   hand corner of normalized device coordinates, (1, -1), should map to the
   lower right hand corner of the viewport, (vpWidth, vpHeight). Another way to put this is,
   the horizontal NDC interval (-1, 1) maps to the horizontal viewport interval (0, vpWidth). and
   the vertical NDC interval (-1, 1) maps the the vertical viewport interval (vpHeight, 0).

   For each vertex in a line segment with NDC (x_ndc, y_ndc),
      x_vp = (vpWidth /2.0)*(x_ndc + 1.0)
      y_vp = (vpHeight/2.0)*(1.0 - y_ndc)
*/

public class P4_NDC2VP
{
   /**
      For each line segment in the model, transform each
      vertex from normalized device coordinates to
      viewport coordinates.
   */
   public static void p4_NDC2VP(Model model, int vpWidth, int vpHeight)
   {
      for (LineSegment ls : model.lineSegmentList)
      {
         // The "normalized device coordinates" are all between -1 and +1.
         // The horizontal viewport coordinate should be between 0 and vpWidth.
         // The vertical viewport coordinate should be between 0 and vpHeight.
         // So x_vp = (vpWidth /2.0)*(x_ndc + 1.0)
         //    y_vp = (vpHeight/2.0)*(1.0 - y_ndc)
         ls.v[0].x = (vpWidth /2.0) * (ls.v[0].x + 1.0);
         ls.v[0].y = (vpHeight/2.0) * (1.0 - ls.v[0].y);
         ls.v[1].x = (vpWidth /2.0) * (ls.v[1].x + 1.0);
         ls.v[1].y = (vpHeight/2.0) * (1.0 - ls.v[1].y);
      }
   }
}