/**
   A simple demonstration of loading and drawing a GRS file.

   The basic structure of a GRS file is:
   1.) A number of comment lines, followed by a line starting with an asterisk, '*'.
   2.) The "extent" of the figure in world coordinates (left, top, right, bottom).
   3.) The number of line-strips (i.e, polylines) in the figure.
   4.) The list of line-strips. Each line-strip starts with the number of vertices in the
       line-strip, followed by the (x, y) world coordinates for each vertex.
*/
import java.util.Scanner;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.FileNotFoundException;

public class GRSModel extends Model
{
   // the figure's extents (bounding box)
   public double left   = 0.0;
   public double top    = 0.0;
   public double right  = 0.0;
   public double bottom = 0.0;
   public int numLineStrips = 0;

   public GRSModel(File grsFile)
   {
      super();

      // Open the GRS file.
      String grsName = null;
      FileInputStream fis = null;
      try
      {
         grsName = grsFile.getCanonicalPath();
         fis = new FileInputStream( grsFile );
      }
      catch (FileNotFoundException e)
      {
         e.printStackTrace(System.err);
         System.err.printf("ERROR! Could not open GRS file: %s\n", grsName);
         System.exit(-1);
      }
      catch (IOException e)
      {
         e.printStackTrace(System.err);
         System.err.printf("ERROR! Could not open GRS file: %s\n", grsName);
         System.exit(-1);
      }

      Scanner scanner = new Scanner(fis);

      // Get the geometry from the GRS file.
      try
      {
         // skip over the comment lines
         String line = scanner.nextLine();
         while ( ! line.startsWith("*") )
         {
            //System.err.println(line);
            line = scanner.nextLine();
         }

         // read the figure extents
         this.left   = scanner.nextDouble();
         this.top    = scanner.nextDouble();
         this.right  = scanner.nextDouble();
         this.bottom = scanner.nextDouble();

         // read the number of line-strips
         this.numLineStrips = scanner.nextInt();

         // read each line-strip
         for(int j = 0; j < this.numLineStrips; j++)
         {
            // read the number of vertices in this line-strip
            int numVertices = scanner.nextInt();

            // put this line-strip in the Model object
            double x = scanner.nextDouble();  // read the first vertex in the line-strip
            double y = scanner.nextDouble();
            Vertex v0 = new Vertex(x, y, 0);
            for (int i = 1; i < numVertices; i++)
            {
               // read the next world coordinate pair
               x = scanner.nextDouble();
               y = scanner.nextDouble();
               Vertex v1 = new Vertex(x, y, 0);
               // create a new LineSegment in the Model
               this.addLineSegment( new LineSegment(v0, v1) );
               v0 = v1;
            }
         }
         fis.close();
      }
      catch (IOException e)
      {
         e.printStackTrace(System.err);
         System.err.printf("ERROR! Could not read GRS file: %s\n", grsName);
         System.exit(-1);
      }
   }
}//GRSModel