/**
   This torus is the surface of revolution generated by revolving
   the circle in the xy-plane with radius r2 and center (r1, 0, 0)
   around the y-axis. We are assuming that r1 > r2.

   r1 radius of the circle of revolution
   r2 radius of the cros section circle

   k number of line segments in the circle of revolution
   n number of line segments in the cross section circle


   Here are parametric equations for the circle in the xy-plane
   with radius r2 and center (r1,0, 0).
      x(theta) = r1 + r2*cos(theta)
      y(theta) =      r2*sin(theta)
      z(theta) = 0
   Here is a 3D rotation matrix that revolves around the y-axis
   by phi radians.
      [cos(phi)   0   sin(phi)]
      [  0        1     0     ]
      [sin((phi)  0  -cos(phi)]
   If we multiply the rotation matrix with the circle parameterization,
   we get a parameterization of the torus.
      [cos(phi)   0   sin(phi)] [r1 + r2*cos(theta)]
      [  0        1     0     ] [     r2*sin(theta)]
      [sin((phi)  0  -cos(phi)] [     0            ]

      = ( r1*cos(phi) + r2*cos(phi)*cos(theta).
          r2*sin(theta),
          r1*sin(phi) + r2*sin(phi)*cos(theta) )

      = ( (r1 + r2*cos(theta))*cos(phi),
           r2*sin(theta),
          (r1 + r2*cos(theta))*sin(phi) )
   See
     http://en.wikipedia.org/wiki/Torus#Geometry
*/

public class TorusModel extends Model
{
   public TorusModel(int k, int n, double r1, double r2)
   {
      super();

      // Create the torus's geometry.

      // An array of vertices to be used to create line segments.
      Vertex[][] v = new Vertex[n][k];

      // Create all the vertices.
      for (int j = 0; j < k; j++) // choose a rotation around the y-axis
      {
         double c1 = Math.cos(j*(2.0*Math.PI)/k);
         double s1 = Math.sin(j*(2.0*Math.PI)/k);
         for (int i = 0; i < n; i++)  // go around a cross section circle
         {
            double c2 = Math.cos(i*(2.0*Math.PI)/n);
            double s2 = Math.sin(i*(2.0*Math.PI)/n);
            v[i][j] = new Vertex( (r1 + r2*c2)*c1,
                                   r2*s2,
                                  (r1 + r2*c2)*s1);
         }
      }

      LineSegment ls;
      // Create the vertical cross-section circles.
      for (int j = 0; j < k; j++) // choose a rotation around the y-axis
      {
         for (int i = 0; i < n-1; i++)  // go around a cross section circle
         {
            ls = new LineSegment( v[i][j], v[i+1][j] );
            this.addLineSegment( ls );
         }
         // close the circle
         ls = new LineSegment( v[n-1][j], v[0][j] );
         this.addLineSegment( ls );
      }

      // Create all the horizontal circles around the torus.
      for (int i = 0; i < n; i++)  // go around the cross section circle
      {
         for (int j = 0; j < k-1; j++)  // go around a horizontal circle
         {
            ls = new LineSegment( v[i][j], v[i][j+1] );
            this.addLineSegment( ls );
         }
         // close the circle
         ls = new LineSegment( v[i][k-1], v[i][0] );
         this.addLineSegment( ls );
      }
   }
}//TorusModel