/*

*/
package framebuffer;

import java.awt.*;
import java.awt.event.*;
import java.awt.image.MemoryImageSource;
import javax.swing.JFrame;
import javax.swing.JPanel;

/**
   This class allows our renderer to be interactive. This class is
   an interface between our renderer and the Java GUI system.

   This class is meant to be sub classed by classes that implement
   event listeners. The event listeners provide the interactivity.

   Each instance of InteractiveFrame has a reference to a FrameBuffer
   object (with the same dimensions as the JFrame). When a GUI event
   happens, one of the implemented event listeners will update the
   framebuffer by having the renderer render the scene into the
   framebuffer. When the renderer is done updating the framebuffer,
   this class will pass the framebufer's pixel data to the Graphics
   context of this JFrame. This will display the framebuffer's contents
   in the JFrame's window.

   This is a resizeable window. When the window resizes, the FrameBuffer
   object needs to also resize. But FrameBuffer objects cannot be resized.
   So this class instantiates a new FrameBuffer object each time the window
   is resized.
*/
@SuppressWarnings("serial")
public class InteractiveFrame extends JFrame implements
   FocusListener,  ComponentListener, WindowListener,
   ActionListener, ItemListener,      AdjustmentListener,
   TextListener,   MouseListener,     MouseMotionListener,
   KeyListener
{
   private FrameBuffer fb;
   private MemoryImageSource source;
   private Image img;

   /**
      @param title  title for the JFrame window
      @param width  width of the JFrame window
      @param height height of the JFrame window
   */
   public InteractiveFrame(String title, int width, int height)
   {
      super(title);
      setLocationByPlatform(true);
      setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);

      setSize(width, height);
      setVisible(true);
      java.awt.Insets insets = getInsets();
      //System.err.println(""+insets.left+" "+insets.right+" "+insets.top+" "+insets.bottom);
      setSize(width+insets.left+insets.right, height+insets.top+insets.bottom);

      this.createFrameBuffer(width, height);

      // Give this JFrame a JPanel.
      setContentPane(new JPanel()
      {
         @Override
         protected void paintComponent(Graphics g)
         {
            // Send the new pixel data to the ImageProducer.
            source.newPixels(0, 0, fb.getWidthFB(), fb.getHeightFB());

            Graphics2D g2 = (Graphics2D)g.create();
            g2.drawImage(img, 0, 0, this);
            g2.dispose();
         }
      });
   }


   /**
      @return a reference to the FrameBuffer owned by this window
   */
   public FrameBuffer getFrameBuffer()
   {
      return this.fb;
   }


   /**
      Change the FrameBuffer being used as the source for the Image painted on this JFrame's JPanel.

      This will usually be in response to a call to the componentResized() event listener.

      @param width  new width for this JFrame window
      @param height new height for this JFrame window
   */
   public void createFrameBuffer(int width, int height)
   {
      this.fb = new FrameBuffer(width, height);

      // Use the framebuffer as the source for an Image.
      this.source = new MemoryImageSource(this.fb.getWidthFB(), this.fb.getHeightFB(),
                                          this.fb.pixel_buffer, 0, this.fb.getWidthFB());
      this.source.setAnimated(true);
      this.img = createImage(this.source);
   }


   // implement the ActionListener interface
   @Override public void actionPerformed(ActionEvent e){}
   // implement the ItemListener interface
   @Override public void itemStateChanged(ItemEvent e){}
   // implement the AdjustmentListener interface
   @Override public void adjustmentValueChanged(AdjustmentEvent e){}
   // implement the TextListener interface
   @Override public void textValueChanged(TextEvent e){}
   // implement the KeyListener interface (three methods)
   @Override public void keyPressed(KeyEvent e){}
   @Override public void keyReleased(KeyEvent e){}
   @Override public void keyTyped(KeyEvent e){}
   // implement the MouseListener interface (five methods)
   @Override public void mouseClicked(MouseEvent e){}
   @Override public void mousePressed(MouseEvent e){}
   @Override public void mouseReleased(MouseEvent e){}
   @Override public void mouseEntered(MouseEvent e){}
   @Override public void mouseExited(MouseEvent e){}
   // implement the MouseMotionListener interface
   @Override public void mouseDragged(MouseEvent e){}
   @Override public void mouseMoved(MouseEvent e){}
   // implement the ComponentListener interface (four methods)
   @Override public void componentMoved(ComponentEvent e){}
   @Override public void componentHidden(ComponentEvent e){}
   @Override public void componentResized(ComponentEvent e){}
   @Override public void componentShown(ComponentEvent e){}
   // implement the FocusListener interface
   @Override public void focusGained(FocusEvent e){}
   @Override public void focusLost(FocusEvent e){}
   // implement the WindowListener interface (seven methods)
   @Override public void windowClosed(WindowEvent e){}
   @Override public void windowDeiconified(WindowEvent e){}
   @Override public void windowIconified(WindowEvent e){}
   @Override public void windowActivated(WindowEvent e){}
   @Override public void windowDeactivated(WindowEvent e){}
   @Override public void windowOpened(WindowEvent e){}
   @Override public void windowClosing(WindowEvent e){}
}
