/*

*/
package pipeline;
import scene.*;

/**
   Clip any (projected) line segment that sticks out of the standard view
   rectangle in the view plane. Interpolate vertex color from the clipped off
   vertex to the new vertex.

   This clipping algorithm is a simplification of the Liang-Barsky Parametric
   Line Clipping algorithm.

   This algorithm assumes that all LineSegment objects have been projected onto
   the camera's view plane, z = -1. This algorithm also assumes that the camera's
   view rectangle has been "normalized" to a standard clipping rectangle, with
      -1 <= x <= +1  and
      -1 <= y <= +1,
   and that the coordinates of each vertex of each line segment have been
   converted to normalized device coordinates.

   If a line segment's projected and normalized vertex has an x or y coordinate
   with absolute value greater than 1, then that vertex "sticks out" of the
   standard (normalized) clipping rectangle. This algorithm will clip the
   line segment so that both of the line segment's vertices are within the
   standard clipping rectangle with -1 <= x <= +1 and -1 <= y <= +1.


   Here is an outline of the clipping algorithm.

   Recursively process each line segment, using the following steps.

     1) Test if the line segment no longer needs to be clipped, i.e., both of
        its vertices are within the normalized clipping rectangle. If this is
        the case, then return true.

     2) Test if the line segment should be "trivially rejected". A line segment
        is "trivially rejected" if it is on the wrong side of any of the four
        lines that bound the normalized clipping rectangle (i.e., the four lines
        x = 1, x = -1, y = 1, y = -1). If so, then return false (so the line
        segment will not be rasterized into the framebuffer).

        Notice that a line like the following one is trivially rejected because it
        is on the "wrong" side of the line x=1.

                           x=1
                            |                      v1
                            |                    /
                 +----------+                  /
                 |          |                /
                 |          |              /
                 |          |            /
                 |          |          /
                 |          |        /
                 +----------+      /
                            |    /
                            |  v0

        But the following line is NOT trivially rejected because, even though it
        is completely outside of the normalized clipping region, this line is not
        entirely on the wrong side of any one of the four lines x = 1, x = -1, y = 1,
        or y = -1. The line below will get clipped at least one time (either on the
        line x = 1 or the line y = -1) before it is (recursively) a candidate for
        "trivial rejection". Notice that the line below could even be clipped twice,
        first on y = 1, then on x = 1, before it can be trivially rejected (by being
        on the wrong side of y = -1).

                           x=1
                            |                      v1
                            |                    /
                 +----------+                  /
                 |          |                /
                 |          |              /
                 |          |            /
                 |          |          /
                 |          |        /
                 +----------+      /
                            |    /
                            |  /
                             /
                           /
                        v0  |
                            |

     3) If the line segment has been neither accepted nor rejected, then it needs
        to be clipped. So we test the line segment against each of the four clipping
        lines, x = 1, x = -1, y = 1, and y = -1, to determine if the line segment
        crosses one of those lines. We clip the line segment against the first line
        which we find that it crosses. Then we recursively clip the resulting clipped
        line segment. (Notice that we only clip against the first clipping line which
        the segment is found to cross. We do not continue to test against the other
        clipping lines. This is because it may be the case, after just one clip, that
        the line segment is now a candidate for trivial accept or reject. So rather
        than test the line segment against several more clipping lines (which may be
        useless tests) it is more efficient to recursively clip the line segment,
        which will then start with the trivial accept or reject tests.)

        When we clip a line segment against a clipping line, it is always the case that
        one endpoint of the line segment is on the "right" side of the clipping line and
        the other endpoint of the line segment is on the "wrong" side of the clipping
        line. In the following picture, assume that v0 is on the "wrong" side of the
        clipping line (x = 1) and v1 is on the "right" side. So v0 needs to be clipped
        off the line segment and replaced by a new endpoint.

                             x=1
                              |
                        v1    |
                          \   |
                            \ |
                              \
                              | \
                              |   \
                              |     v0

        Represent points p(t) on the line segment between v0 and v1 with the
        following parametric equation.
                  p(t) = (1-t) * v0 + t * v1  with  0 <= t <= 1
        Notice that this equation parameterizes the line segment starting with v0
        at t=0 (on the "wrong side") and ending with v1 at t=1 (on the "right side").
        We need to find the value of t when the line segment crosses the clipping
        line x = 1. Let v0=(x0, y0) and let v1=(x1, y1). Then the above parametric
        equation becomes the two component equations
                 x(t) = (1-t) * x0 + t * x1,
                 y(t) = (1-t) * y0 + t * y1,  with  0 <= t <= 1.
        Since the clipping line in this example is x = 1, we need to solve the
        equation x(t) = 1 for t. So we need to solve
                  (1-t) * x0 + t * x1 = 1
        for t. Here are a few algebra steps.
                  x0 - t * x0 + t * x1 = 1
                  x0 + t * (x1 - x0) = 1
                       t * (x1 - x0) = 1 - x0
                       t = (1 - x0)/(x1 - x0)
        We get similar equations for t if we clip against the other clipping lines
        (x = -1, y = 1, or y = -1) and we assume that v0 is on the "wrong side" and
        v1 is on the "right side". With the above value for t, the new point p(t)
        that replaces v0 can easily be computed.

                             x=1
                              |
                        v1    |
                          \   |
                            \ |
                              v0=p( (1 - x0)/(x1 - x0) )
                              |
                              |
                              |

         Finally, the new line segment between v1 and the new v0 is recursively
         clipped so that it can be checked once again to see if it can be trivially
         accepted, trivially rejected, or clipped again.
*/
public class Clip
{
   /**
      If the line segment sticks out of the normalized view rectangle,
      then clip it so that it is contained in the normalized view rectangle.
   */
   public static boolean clip(LineSegment ls)
   {
      // make local copies of several values
      Vertex v0 = ls.v[0],  v1 = ls.v[1];
      double x0 = v0.x,     x1 = v1.x;
      double y0 = v0.y,     y1 = v1.y;

      // 1. Check for trivial accept.
      if ( ! ( Math.abs( x0 ) > 1
            || Math.abs( y0 ) > 1
            || Math.abs( x1 ) > 1
            || Math.abs( y1 ) > 1 ) )
      {
         //System.err.println("Trivial accept.");
         return true;
      }
      // 2. Check for trivial delete.
      else if ( (x0 >  1 && x1 >  1)   // to the right of the line x = 1
             || (x0 < -1 && x1 < -1)   // to the left of the line x = -1
             || (y0 >  1 && y1 >  1)   // above the line y = 1
             || (y0 < -1 && y1 < -1) ) // below the line y = -1
      {
         //System.err.println("Trivial delete.");
         return false;
      }
      // 3. Need to clip this line segment.
      if (x0 > 1 || x1 > 1)  // ls crosses the line x = 1
      {
         if (x1 > 1)  // clip off v1
         {
            // create a new Vertex between v[0] and v[1]
            Vertex v_new = interpolateNewVertex(v0, v1, 1);
            // modify the LineSegment to contain v[0] and the new Vertex
            ls.v[1] = v_new;
         }
         else // (x0 > 1)  // clip off v0
         {
            // create a new Vertex between v[1] and v[0]
            Vertex v_new = interpolateNewVertex(v1, v0, 1);
            // modify the LineSegment to contain v[1] and the new Vertex
            ls.v[0] = v_new;
         }
      }
      else if (x0 < -1 || x1 < -1)  // ls crosses the line x = -1
      {
         if (x1 < -1)  // clip off v1
         {
            // create a new Vertex between v0 and v1
            Vertex v_new = interpolateNewVertex(v0, v1, 2);
            // modify the LineSegment to contain v0 and the new Vertex
            ls.v[1] = v_new;
         }
         else // (x0 < -1)  // clip off v0
         {
            // create a new Vertex between v1 and v0
            Vertex v_new = interpolateNewVertex(v1, v0, 2);
            // modify the LineSegment to contain v1 and the new Vertex
            ls.v[0] = v_new;
         }
      }
      else if (y0 > 1 || y1 > 1)  // ls crosses the line y = 1
      {
         if (y1 > 1)  // clip off v1
         {
            // create a new Vertex between v0 and v1
            Vertex v_new = interpolateNewVertex(v0, v1, 3);
            // modify the LineSegment to contain v0 and the new Vertex
            ls.v[1] = v_new;
         }
         else // (y0 > 1)  // clip off v0
         {
            // create a new Vertex between v1 and v0
            Vertex v_new = interpolateNewVertex(v1, v0, 3);
            // modify the LineSegment to contain v1 and the new Vertex
            ls.v[0] = v_new;
         }
      }
      else if (y0 < -1 || y1 < -1)  // ls crosses the line y = -1
      {
         if (y1 < -1)  // clip off v1
         {
            // create a new Vertex between v0 and v1
            Vertex v_new = interpolateNewVertex(v0, v1, 4);
            // modify the LineSegment to contain v0 and the new Vertex
            ls.v[1] = v_new;
         }
         else // (y0 < -1)  // clip off v0
         {
            // create a new Vertex between v1 and v0
            Vertex v_new = interpolateNewVertex(v1, v0, 4);
            // modify the LineSegment to contain v1 and the new Vertex
            ls.v[0] = v_new;
         }
      }
      else
      {  // should never get here
         System.err.println("Clipping error!");
         System.exit(-1);
      }
      return clip(ls);  // recursively clip this line segment again
   }


   /**
      This method takes in two vertices, one that is on the "right" side
      of a clipping line and the other that is on the "wrong" side of the
      clipping line, and an integer which specifies which clipping line
      to use, where
         eqn_number == 1  means the clipping line x = 1
         eqn_number == 2  means the clipping line x =-1
         eqn_number == 3  means the clipping line y = 1
         eqn_number == 4  means the clipping line y =-1

      This method returns the vertex that is the intersection point
      between the given line segment and the given clipping line.

      This method solves for the value of t for which the parametric
      equation
                  p(t) = (1-t) * v_outside + t * v_inside
      intersects the given clipping line (notice that the equation
      is parameterized so that we move from the outside vertex towards
      the inside vertex as t increases from 0 to 1). The solved for
      value of t is then plugged into the parametric formula to get
      the coordinates of the intersection point.
   */
   private static Vertex interpolateNewVertex(Vertex v_inside,
                                              Vertex v_outside,
                                              int eqn_number)
   {
      //System.err.println("Create new vertex.");
      // make local copies of several values
      double vix =  v_inside.x; // "i" for "inside"
      double viy =  v_inside.y;
      double vox = v_outside.x; // "o" for "outside"
      double voy = v_outside.y;
      // interpolate between v_outside and v_inside
      double t = 0.0;
      if (1 == eqn_number)            // clip to x = 1
         t = (1 - vox) / (vix - vox);
      else if (2 == eqn_number)       // clip to x = -1
         t = (-1 - vox) / (vix - vox);
      else if (3 == eqn_number)       // clip to y = 1
         t = (1 - voy) / (viy - voy);
      else if (4 == eqn_number)       // clip to y = -1
         t = (-1 - voy) / (viy - voy);

      Vertex v_new = new Vertex();

      // use the value of t to interpolate the coordinates of the new vertex
      v_new.x = (1-t) * vox + t * vix;
      v_new.y = (1-t) * voy + t * viy;
      // use the value of t to interpolate the color of the new vertex
      v_new.r = (1-t) * v_outside.r + t * v_inside.r;
      v_new.g = (1-t) * v_outside.g + t * v_inside.g;
      v_new.b = (1-t) * v_outside.b + t * v_inside.b;

      return v_new;
   }

}
