/*

*/
package pipeline;
import scene.*;
import framebuffer.*;

/**
    This renderer takes as its input a Scene data structure and
    a FrameBuffer data structure. This renderer mutates the
    FrameBuffer's current viewport so that it is filled in with
    the rendered image of the scene represented by the Scene object.
<p>
    This implements our seventh rendering pipeline. This renderer
    implements projection as a matrix transformation followed by
    perspective division, which is a new pipeline stage. (There
    are now six pipeline stages.)
*/
public class Pipeline
{
   public static boolean doAntialiasing = false;
   public static boolean debug = false;
   public static boolean debug_visible = false;

   /**
      Mutate the FrameBuffer's current viewport so that it holds
      the rendered image of the Scene object.

      @param scene Scene object to render
      @param fb    FrameBuffer to hold rendered image of the scene
   */
   public static void render(Scene scene, FrameBuffer fb)
   {
      if ( debug || debug_visible )
         render_debug(scene, fb);
      else
         render_(scene, fb);
   }


   /**
      Renderer a scene without any debugging information.

      @param scene Scene object to render
      @param fb    FrameBuffer to hold rendered image of the scene
   */
   private static void render_(Scene scene, FrameBuffer fb)
   {
      // For every Model in the Scene, render the model
      // and every sub model in the Model.
      for (Model model : scene.modelList)
      {
         render_model(scene, model, new Matrix(1,1,1), fb);
      }
   }


   /**
      Recursively renderer a model without any debugging information.

      @param scene the Scene that we are rendering
      @param model the current Model object to recursively render
      @param ctm   current transformation matrix
      @param fb    FrameBuffer to hold rendered image of the model
   */
   private static void render_model(Scene scene, Model model, Matrix ctm, FrameBuffer fb)
   {
      // Update the current transformation matrix.
      ctm = ctm.times( model.modelMatrix );

      // Render every LineSgement in the Model.
      for (LineSegment ls : model.lineSegmentList)
      {
         // Make a deep copy of the line segment.
         LineSegment ls2 = new LineSegment(ls);

         // 1. Apply the current model-to-camera coordinate transformation.
         Model2Camera.model2camera(ls2, ctm);

         // 2. Apply the projection transformation.
         Project.project(ls2, scene.camera.projMatrix);

         // 3. Apply the perspective division operation.
         PerspectiveDivision.perspectiveDivide(ls2);

         // 4. Clip the line segment to the standard view rectangle.
         if ( Clip.clip(ls2) )
         {
            // 5. Apply the viewport transformation.
            Viewport.viewport(ls2, fb);

            // 6. Rasterize the line segment into pixels.
            if ( doAntialiasing )
            {
               RasterizeAntialias.rasterize(ls2, fb);
            }
            else
            {
               Rasterize.rasterize(ls2, fb);
            }
         }
      }

      // Recursively render every sub model of this model.
      for (Model m : model.nestedModels)
      {
         render_model(scene, m, ctm, fb);
      }
   }



   /**
      Render a scene with quite a bit of debugging information.

      @param scene Scene object to render
      @param fb    FrameBuffer to hold rendered image of the scene
   */
   private static void render_debug(Scene scene, FrameBuffer fb)
   {
      System.out.println("=== Render Scene ===================================");

      // For every Model in the Scene, render the model
      // and every sub model in the Model.
      for (Model model : scene.modelList)
      {
         System.out.println("=== Render Model ===================================");
         render_model_debug(scene, model, new Matrix(1,1,1), fb);
         System.out.println("=== End Model ======================================");
      }

      System.out.println("=== End Scene ===================================");
   }


   /**
      Recursively renderer a model with quite a bit of debugging information.

      @param scene the Scene that we are rendering
      @param model the current Model object to recursively render
      @param ctm   current transformation matrix
      @param fb    FrameBuffer to hold rendered image of the model
   */
   private static void render_model_debug(Scene scene, Model model, Matrix ctm, FrameBuffer fb)
   {
      // Update the current transformation matrix.
      ctm = ctm.times( model.modelMatrix );

      // Render every LineSgement in the Model.
      for (LineSegment ls : model.lineSegmentList)
      {
         StringBuilder result = new StringBuilder();

         // Make a deep copy of the line segment.
         LineSegment ls2 = new LineSegment(ls);

         // log interesting information to standard output
         logLineSegment("0. Model", ls2, result);

         // 1. Apply the current model-to-camera coordinate transformation.
         Model2Camera.model2camera(ls2, ctm);

         // log interesting information to standard output
         logLineSegment("1. Camera", ls2, result);

         // 2. Apply the projection transformation.
         Project.project(ls2, scene.camera.projMatrix);

         // log interesting information to standard output
         logLineSegment("2. Projected", ls2, result);

         // 3. Apply the perspective division operation.
         PerspectiveDivision.perspectiveDivide(ls2);

         // log interesting information to standard output
         logLineSegment("3. Divided", ls2, result);

         // 4. Clip the line segment to the standard view rectangle.
         if ( Clip.clip(ls2) )
         {
            // log interesting information to standard output
            logLineSegment("4. Clipped", ls2, result);

            // 5. Apply the viewport transformation.
            Viewport.viewport(ls2, fb);

            // log interesting information to standard output
            logLineSegment("5. Viewport", ls2, result);

            // 6. Rasterize the line segment into pixels.
            if ( doAntialiasing )
            {
               RasterizeAntialias.rasterize(ls2, fb);
            }
            else
            {
               Rasterize.rasterize(ls2, fb);
            }
            if ( debug_visible ) System.out.println( result );
         }
         if ( ! debug_visible ) System.out.println();
      }

      // Recursively renderer every sub model in this model.
      for (Model m : model.nestedModels)
      {
         System.out.println("====== Render Sub Model ========================");
         render_model_debug(scene, m, ctm, fb);
         System.out.println("====== End Sub Model ===========================");
      }
   }


   /**
      This method either outputs a String representation of the given
      LineSegment or it accumulates the String representation in the
      accumulator variable (for outputting later).

      @param stage name for the pipeline stage
      @param ls LineSegment whose string representation is to be appended to the accumulator
      @param accumulator accumulates the string representations for all the line segments in this stage
   */
   private static void logLineSegment(String stage, LineSegment ls, StringBuilder accumulator)
   {
      if ( debug_visible )
         accumulator.append( stage + " " + ls.toString() );
      else
         System.out.print( stage + " " + ls.toString() );
   }
}
