/*

*/
package pipeline;
import scene.*;
import framebuffer.*;

import java.awt.Color;

/**
   Rasterize each line segment into pixels in the framebuffer.

   This rasterization algorithm is based on
     "Fundamentals of Computer Graphics", 3rd Edition,
      by Peter Shirley, pages 163-165.
*/
public class Rasterize
{
   public static boolean debug = false;

   /**
      Rasterize each line segment into pixels in the framebuffer.
   */
   public static void rasterize(LineSegment ls, FrameBuffer fb)
   {
      int w = fb.getWidthVP();
      int h = fb.getHeightVP();

      // Get the colors from the two vertices.
      double r0 = ls.v[0].r;
      double g0 = ls.v[0].g;
      double b0 = ls.v[0].b;
      double r1 = ls.v[1].r;
      double g1 = ls.v[1].g;
      double b1 = ls.v[1].b;

      // Round the line segment's two endpoints to the nearest
      // logical pixel. This makes the algorithm a lot simpler,
      // but it can cause a slight, but noticable, shift of the
      // line segment.
      double x0 = Math.round( ls.v[0].x );
      double y0 = Math.round( ls.v[0].y );
      double x1 = Math.round( ls.v[1].x );
      double y1 = Math.round( ls.v[1].y );

      if (Math.abs(y1 - y0) <= Math.abs(x1 - x0)) // if abs(slope) <= 1
      {
         if (x1 < x0) // swap (x0, y0) with (x1, y1)
         {
            double tempX = x0;
            double tempY = y0;
            x0 = x1;
            y0 = y1;
            x1 = tempX;
            y1 = tempY;
            // swap the colors too
            double tempR = r0;
            double tempG = g0;
            double tempB = b0;
            r0 = r1;
            g0 = g1;
            b0 = b1;
            r1 = tempR;
            g1 = tempG;
            b1 = tempB;
         }

         // Compute this line segment's slope.
         double      m = (y1 - y0) / (x1 - x0);
         double slopeR = (r1 - r0) / (x1 - x0);
         double slopeG = (g1 - g0) / (x1 - x0);
         double slopeB = (b1 - b0) / (x1 - x0);

         // In the following loop, as x moves across the logical
         // horizontal pixels, we will compute a y value for each x.
         double y = y0;

         // Rasterize this line segment.
         for (int x = (int)x0; x <= (int)x1; x++, y += m)
         {
            // Interpolate this pixel's color between the two endpoint's colors.
            float r = (float)Math.abs(r0 + slopeR*(x - x0));
            float g = (float)Math.abs(g0 + slopeG*(x - x0));
            float b = (float)Math.abs(b0 + slopeB*(x - x0));
            // We need the Math.abs() because otherwise, we sometimes get -0.0.

            // The value of y will almost always be between
            // two vertical pixel coordinates. By rounding off
            // the value of y, we are choosing the nearest logical
            // vertical pixel coordinate.
            fb.setPixelVP(x - 1, h - (int)Math.round(y), new Color(r, g, b));

            // log interesting information to standard output
            if (debug) logPixel(w, h, x, y, r, g, b);

            // Advance (x,y) to the next pixel. Since delta_x = 1, we need delta_y = m.
         }
      }
      else  // abs(slope) > 1
      {
         if (y1 < y0) // swap (x0, y0) with (x1, y1)
         {
            double tempX = x0;
            double tempY = y0;
            x0 = x1;
            y0 = y1;
            x1 = tempX;
            y1 = tempY;
            // swap the colors too
            double tempR = r0;
            double tempG = g0;
            double tempB = b0;
            r0 = r1;
            g0 = g1;
            b0 = b1;
            r1 = tempR;
            g1 = tempG;
            b1 = tempB;
         }

         // Compute this line segment's slope.
         double      m = (x1 - x0) / (y1 - y0);
         double slopeR = (r1 - r0) / (y1 - y0);
         double slopeG = (g1 - g0) / (y1 - y0);
         double slopeB = (b1 - b0) / (y1 - y0);

         // In the following loop, as y moves across the logical
         // vertical pixels, we will compute a x value for each y.
         double x = x0;

         // Rasterize this line segment.
         for (int y = (int)y0; y <= (int)y1; x += m, y++)
         {
            // Interpolate this pixel's color between the two endpoint's colors.
            float r = (float)Math.abs(r0 + slopeR*(y - y0));
            float g = (float)Math.abs(g0 + slopeG*(y - y0));
            float b = (float)Math.abs(b0 + slopeB*(y - y0));
            // We need the Math.abs() because otherwise, we sometimes get -0.0.

            // The value of x will almost always be between
            // two horizontal pixel coordinates. By rounding off
            // the value of x, we are choosing the nearest logical
            // horizontal pixel coordinate.
            fb.setPixelVP((int)Math.round(x) - 1, h - y, new Color(r, g, b));

            // log interesting information to standard output
            if (debug) logPixel(w, h, x, y, r, g, b);

            // Advance (x,y) to the next pixel. Since delta_y = 1, we need delta_x = m.
         }
      }
   }


   private static void logPixel(int w, int h, int x, double y, double r, double g, double b)
   {
      System.out.printf("[w=%d, h=%d]  x=%d, y=%f, r=%f, g=%f, b=%f\n", w, h, x, y, r, g, b);
   }

   private static void logPixel(int w, int h, double x, int y, double r, double g, double b)
   {
      System.out.printf("[w=%d, h=%d]  x=%f, y=%d, r=%f, g=%f, b=%f\n", w, h, x, y, r, g, b);
   }
}
