/*

*/
package pipeline;
import scene.*;
import framebuffer.*;

import java.awt.Color;

/**
   Rasterize each line segment into pixels in the framebuffer
   and antialias the lines at the same time.

   This is a fairly simplistic antialiasing algorithm.

*/
public class RasterizeAntialias
{
   public static boolean debug = false;

   /**
      Rasterize each line segment into pixels in the framebuffer.
   */
   public static void rasterize(LineSegment ls, FrameBuffer fb)
   {
      // Get the viewport's background color.
      Color bg = fb.getBgColorVP();

      int w = fb.getWidthVP();
      int h = fb.getHeightVP();

      // Get the colors from the two vertices.
      double r0 = ls.v[0].r;
      double g0 = ls.v[0].g;
      double b0 = ls.v[0].b;
      double r1 = ls.v[1].r;
      double g1 = ls.v[1].g;
      double b1 = ls.v[1].b;

      // Round the line segment's two endpoints to the nearest
      // logical pixel. This makes the algorithm a lot simpler,
      // but it can cause a slight, but noticable, shift of the
      // line segment.
      double x0 = Math.round( ls.v[0].x );
      double y0 = Math.round( ls.v[0].y );
      double x1 = Math.round( ls.v[1].x );
      double y1 = Math.round( ls.v[1].y );

      if (Math.abs(y1 - y0) <= Math.abs(x1 - x0)) // if abs(slope) <= 1
      {
         if (x1 < x0) // swap (x0, y0) with (x1, y1)
         {
            double tempX = x0;
            double tempY = y0;
            x0 = x1;
            y0 = y1;
            x1 = tempX;
            y1 = tempY;
            // swap the colors too
            double tempR = r0;
            double tempG = g0;
            double tempB = b0;
            r0 = r1;
            g0 = g1;
            b0 = b1;
            r1 = tempR;
            g1 = tempG;
            b1 = tempB;
         }

         // Compute this line segment's slope.
         double      m = (y1 - y0) / (x1 - x0);
         double slopeR = (r1 - r0) / (x1 - x0);
         double slopeG = (g1 - g0) / (x1 - x0);
         double slopeB = (b1 - b0) / (x1 - x0);

         // Rasterize this line segment.
         // In the following loop, as x moves across the logical
         // horizontal pixels, we will compute a y value for each x.
         double y = y0;
         for (int x = (int)x0; x < (int)x1; x++, y += m)
         {
            // Interpolate this pixel's color between the two endpoint's colors.
            float r = (float)Math.abs(r0 + slopeR*(x - x0));
            float g = (float)Math.abs(g0 + slopeG*(x - x0));
            float b = (float)Math.abs(b0 + slopeB*(x - x0));
            // We need the Math.abs() because otherwise, we sometimes get -0.0.

            // Check if y is on the bottom or top edge of the pixel coordinate space.
            if ( y <= 1 || h <= y )
            {
               // Set this pixel in the framebuffer.
               fb.setPixelVP(x - 1, h - (int)Math.round(y), new Color(r, g, b));

               // log interesting information to standard output
               if (debug) logPixel(w, h, x, y, r, g, b);
            }
            else // y must be between two logical vertical pixel coordinates
            {
               // Let y_low and y_hi be the logical vertical pixel
               // coordinates that bracket around y.
               int y_low = (int)y;                // the integer part of y
               int y_hi  = y_low + 1;
               // Let weight be the fractional part of y. We will use
               // weight to determine how much emphasis to place on
               // each of the two pixels that bracket y.
               float weight = (float)(y - y_low);

               // Interpolate colors for the low and high pixels.
               // The smaller weight is, the closer y is to the lower
               // pixel, so we give the lower pixel more emphasis when
               // weight is small.
               float r_low = (float)((1 - weight) * r + weight * (bg.getRed()/255.0));
               float g_low = (float)((1 - weight) * g + weight * (bg.getGreen()/255.0));
               float b_low = (float)((1 - weight) * b + weight * (bg.getBlue()/255.0));
               float r_hi  = (float)(weight * r + (1 - weight) * (bg.getRed()/255.0));
               float g_hi  = (float)(weight * g + (1 - weight) * (bg.getGreen()/255.0));
               float b_hi  = (float)(weight * b + (1 - weight) * (bg.getBlue()/255.0));

               // Set this (antialiased) pixel in the framebuffer.
               fb.setPixelVP(x - 1, h - y_low, new Color(r_low, g_low, b_low));
               fb.setPixelVP(x - 1, h - y_hi,  new Color(r_hi,  g_hi,  b_hi));

               // log interesting information to standard output
               if (debug) logPixel(w, h, x, y, r_low, g_low, b_low, r_hi, g_hi, b_hi);
            }
            // Advance (x,y) to the next pixel (delta_x is 1, so delta_y is m).
         }
         // Set the pixel for the (x1,y1) endpoint
         // (we do this separately to avoid roundoff errors).
         fb.setPixelVP((int)x1 - 1, h - (int)y1, new Color((float)r1, (float)g1, (float)b1));
      }
      else  // abs(slope) > 1
      {
         if (y1 < y0) // swap (x0, y0) with (x1, y1)
         {
            double tempX = x0;
            double tempY = y0;
            x0 = x1;
            y0 = y1;
            x1 = tempX;
            y1 = tempY;
            // swap the colors too
            double tempR = r0;
            double tempG = g0;
            double tempB = b0;
            r0 = r1;
            g0 = g1;
            b0 = b1;
            r1 = tempR;
            g1 = tempG;
            b1 = tempB;
         }

         // Compute this line segment's slope.
         double      m = (x1 - x0) / (y1 - y0);
         double slopeR = (r1 - r0) / (y1 - y0);
         double slopeG = (g1 - g0) / (y1 - y0);
         double slopeB = (b1 - b0) / (y1 - y0);

         // Rasterize this line segment.
         // In the following loop, as y moves across the logical
         // vertical pixels, we will compute a x value for each y.
         double x = x0;
         for (int y = (int)y0; y < (int)y1; x += m, y++)
         {
            // Interpolate this pixel's color between the two endpoint's colors.
            float r = (float)Math.abs(r0 + slopeR*(y - y0));
            float g = (float)Math.abs(g0 + slopeG*(y - y0));
            float b = (float)Math.abs(b0 + slopeB*(y - y0));
            // We need the Math.abs() because otherwise, we sometimes get -0.0.

            // Check if x is on the left or right edge of the pixel coordinate space.
            if ( x <= 1 || w <= x )
            {
               // Set this pixel in the framebuffer.
               fb.setPixelVP((int)Math.round(x) - 1, h - y, new Color(r, g, b));

               // log interesting information to standard output
               if (debug) logPixel(w, h, x, y, r, g, b);
            }
            else // x must be between two logical horizontal pixel coordinates
            {
               // x_low and x_hi will be pixel coordinates that bracket around x
               int x_low = (int)x;                // the integer part of x
               int x_hi  = x_low + 1;

               float weight = (float)(x - x_low); // the fractional part of x

               // Interpolate colors for the low and high pixels.
               float r_low = (float)((1 - weight) * r + weight * (bg.getRed()/255.0));
               float g_low = (float)((1 - weight) * g + weight * (bg.getGreen()/255.0));
               float b_low = (float)((1 - weight) * b + weight * (bg.getBlue()/255.0));
               float r_hi  = (float)(weight * r + (1 - weight) * (bg.getRed()/255.0));
               float g_hi  = (float)(weight * g + (1 - weight) * (bg.getGreen()/255.0));
               float b_hi  = (float)(weight * b + (1 - weight) * (bg.getBlue()/255.0));

               // Set this (antialiased) pixel in the framebuffer.
               fb.setPixelVP(x_low - 1, h - y, new Color(r_low, g_low, b_low));
               fb.setPixelVP( x_hi - 1, h - y, new Color(r_hi,  g_hi,  b_hi));

               // log interesting information to standard output
               if (debug) logPixel(w, h, x, y, r_low, g_low, b_low, r_hi, g_hi, b_hi);
            }
            // Advance (x,y) to the next pixel (delta_y is 1, so delta_x is m).
         }
         // Set the pixel for the (x1,y1) endpoint
         // (we do this separately to avoid roundoff errors).
         fb.setPixelVP((int)x1 - 1, h - (int)y1, new Color((float)r1, (float)g1, (float)b1));
      }
   }


   private static void logPixel(int w, int h, int x, double y, double r, double g, double b)
   {
      System.out.printf("[w=%d, h=%d]  x=%d, y=%f, r=%f, g=%f, b=%f\n", w, h, x, y, r, g, b);
   }

   private static void logPixel(int w, int h, double x, int y, double r, double g, double b)
   {
      System.out.printf("[w=%d, h=%d]  x=%f, y=%d, r=%f, g=%f, b=%f\n", w, h, x, y, r, g, b);
   }

   private static void logPixel(int w, int h, int x, double y, double r1, double g1, double b1,
                                                               double r2, double g2, double b2)
   {
      System.out.printf("[w=%d, h=%d]  x=%d, y=%f, r=%f, g=%f, b=%f\n", w, h, x, y, r1, g1, b1);
      System.out.printf("                x=%d, y=%f, r=%f, g=%f, b=%f\n", x, y, r2, g2, b2);
   }

   private static void logPixel(int w, int h, double x, int y, double r1, double g1, double b1,
                                                               double r2, double g2, double b2)
   {
      System.out.printf("[w=%d, h=%d]  x=%f, y=%d, r=%f, g=%f, b=%f\n", w, h, x, y, r1, g1, b1);
      System.out.printf("                x=%f, y=%d, r=%f, g=%f, b=%f\n", x, y, r2, g2, b2);
   }
}
