/*

*/
package pipeline;
import scene.*;
import framebuffer.*;

/**
<pre>
   Transform each line segment's endpoints from viewplane coordinates
   to viewport coordinates so that the view rectangle in the view plane
   with
        -1 <= x <= 1,
        -1 <= y <= 1,
   transforms into a viewport where
       0.5 <= x < w + 0.5
       0.5 <= y < h + 0.5
   where w = number of horizontal pixels in the viewport,
         h = number of vertical pixels in the viewport.

   The goal of this transformation is to put a logical pixel with
   integer coordinates at the center of each square physical pixel.
   The logical pixel with integer coordinates (m, n) represents the
   square pixel with
     m - 0.5 <= x < m + 0.5,
     n - 0.5 <= y < n + 0.5.
   Notice that logical pixel integer coordinates (m.n) have
     1 <= m <= w
     1 <= n <= h.
</pre>
*/
public class Viewport
{
   /**
      Transform the line segment's endpoints from
      viewplane coordinates to viewport coordinates.

      @param ls LineSegment to transform to viewport coordinates
      @param fb FrameBuffer that holds the current viewport
   */
   public static void viewport(LineSegment ls, FrameBuffer fb)
   {
      // Get the viewport dimensions.
      int w = fb.getWidthVP();
      int h = fb.getHeightVP();

      // Transform the line segment's endpoints from
      // viewplane coordinates to viewport coordinates.
      double x0 = 0.5 + w/2.001 * (ls.v[0].x + 1.0);
      double y0 = 0.5 + h/2.001 * (ls.v[0].y + 1.0);
      double x1 = 0.5 + w/2.001 * (ls.v[1].x + 1.0);
      double y1 = 0.5 + h/2.001 * (ls.v[1].y + 1.0);
      // NOTE: Notice the 2.001 fudge factor in the last four equations.
      // This is explained on page 142 of
      //    "Jim Blinn's Corner: A Trip Down The Graphics Pipeline"
      //     by Jim Blinn, 1996, Morgan Kaufmann Publishers.

      // Mutate the LineSegment object so that it refers to new
      // Vertex objects containing the viewport coordinates.
      ls.v[0] = new Vertex( x0, y0, ls.v[0].z, ls.v[0].getColor() );
      ls.v[1] = new Vertex( x1, y1, ls.v[1].z, ls.v[1].getColor() );
   }
}
