/*

*/
package scene;

/**
   This Camera data structure represents a camera located
   at the origin and looking down the negative z-axis.

   This camera has associated to it a "view volume" that
   determines what part of space the camera "sees" in front
   of it when we use the camera to take a picture (that is,
   when we render a scene).

   This camera can "take a picture" two ways, using a
   perspective projection or using a parallel (orthographic)
   projection. Each way of taking a picture has a different
   shape for its view volume.

   For the perspective projection, the view volume is an
   infinitely long pyramid that is formed by the pyramid
   with its apex at the origin and its base in the plane
   z = -1 with edges x = -1, x = +1, y = -1, and y = +1.

   For the orthographic projection, the view volume is an
   infinitely long rectangular cylinder parallel to the
   z-axis and with sides x = -1, x = +1, y = -1, and y = +1
   (an infinite parallelepiped).

   The part of space determined by the view volume is what
   the camera "sees" when we "take a picture" (that is,
   when we render a scene).

   The plane z = -1 is the camera's view plane. The rectangle
   in the view plane with corners (-1, -1, -1) and (+1, +1, -1)
   is the camera's view rectangle. The view rectangle is like
   the film in a real camera, it is where the camera's image
   appears when you take a picture. The contents of the camera's
   view rectangle is what gets rasterized into the framebuffer.
*/
public class Camera
{
   public boolean perspective;  // choose either perspective or parallel projection

   public Matrix projMatrix;


   /**
      The default camera has a perspective projection matrix.
   */
   public Camera()
   {
      this.projPerspective();
   }


   /**
      Set up this camera's view volume as a perspective projection
      of an infinite view pyramid.
   */
   public void projPerspective()
   {
      this.projMatrix = new PerspectiveProjectionMatrix1();

      this.perspective = true;
   }


   /**
      Set up this camera's view volume as a parallel (orthographic) projection
      of an infinite view parallelepiped.
   */
   public void projOrtho()
   {
      this.projMatrix = new OrthographicProjectionMatrix();

      this.perspective = false;
   }


   /**
      For debugging.

      @return String representation of this Camera object
   */
   public String toString()
   {
      String result = "";
      result += "Camera: \n";
      result += "perspective = " + perspective + "\n";
      result += projMatrix + "\n";
      return result;
   }
}
