/*

*/
package scene;

/**
    A Matrix object has four Vector objects.
<p>
    The four Vector objects represent the four column vectors
    of a 4-by-4 matrix (as in a Linear Algebra course).
<p>
    In computer graphics, the points and vectors of 3-dimensional space
    are represented using 4-dimensional homogeneous coordinates.
    So each transformation of 3-dimensional space is represented by
    a 4-by-4 (homogeneous) matrix.
<p>
    A 4-by-4 matrix represents a transformation of 3-dimensional space.
    The most common transformations are translation, rotation, and
    scaling. A 4-by-4 matrix can also represent a projection transformation.
*/
public class Matrix
{
   public Vector v1, v2, v3, v4;  // these are column vectors


   /**
      Construct a default matrix (of all zeros).
   */
   public Matrix()
   {
      this.v1 = new Vector();
      this.v2 = new Vector();
      this.v3 = new Vector();
      this.v4 = new Vector();
   }


   /**
      Construct a translation matrix that translates
      by the given Vector.

      @param v vector to translate by
   */
   public Matrix(Vector v)
   {
      this.v1 = new Vector( 1.0, 0.0, 0.0, 0.0 );
      this.v2 = new Vector( 0.0, 1.0, 0.0, 0.0 );
      this.v3 = new Vector( 0.0, 0.0, 1.0, 0.0 );
      this.v4 = new Vector( v.x, v.y, v.z, 1.0 );
   }


   /**
      Construct a diagonal matrix with the given number
      on the diagonal.

      This a also a uniform scaling matrix.

      @param d the diagonal value for the new matrix
   */
   public Matrix(double d)
   {
      this.v1 = new Vector(   d, 0.0, 0.0, 0.0 );
      this.v2 = new Vector( 0.0,   d, 0.0, 0.0 );
      this.v3 = new Vector( 0.0, 0.0,   d, 0.0 );
      this.v4 = new Vector( 0.0, 0.0, 0.0, 1.0 );
   }


   /**
      Construct a (diagonal) matrix that scales in
      the x, y, and z directions by the given factors.

      @param x scale factor for the x-direction
      @param y scale factor for the y-direction
      @param z scale factor for the z-direction
   */
   public Matrix(double x, double y, double z)
   {
      this.v1 = new Vector(   x, 0.0, 0.0, 0.0 );
      this.v2 = new Vector( 0.0,   y, 0.0, 0.0 );
      this.v3 = new Vector( 0.0, 0.0,   z, 0.0 );
      this.v4 = new Vector( 0.0, 0.0, 0.0, 1.0 );
   }


   /**
      Construct a rotation matrix.

      See
          https://www.opengl.org/sdk/docs/man2/xhtml/glRotate.xml

      @param theta angle (in degrees) to rotate by around the axis vector
      @param x x-component of the axis vector for the rotation
      @param y y-component of the axis vector for the rotation
      @param z z-component of the axis vector for the rotation
   */
   public Matrix(double theta, double x, double y, double z)
   {
      double norm = Math.sqrt(x*x + y*y + z*z);
      double ux = x/norm;
      double uy = y/norm;
      double uz = z/norm;

      double c = Math.cos( (Math.PI/180.0)*theta );
      double s = Math.sin( (Math.PI/180.0)*theta );

      this.v1 = new Vector( ux*ux*(1-c)+c,      uy*ux*(1-c)+(uz*s), uz*ux*(1-c)-(uy*s), 0.0 );
      this.v2 = new Vector( ux*uy*(1-c)-(uz*s), uy*uy*(1-c)+c,      uz*uy*(1-c)+(ux*s), 0.0 );
      this.v3 = new Vector( ux*uz*(1-c)+(uy*s), uy*uz*(1-c)-(ux*s), uz*uz*(1-c)+c,      0.0 );
      this.v4 = new Vector( 0.0,                0.0,                0.0,                1.0 );
   }


   /**
      Construct an arbitrary 4-by-4 matrix with the given column vectors.

      @param v1 1st column vector for the new matrix
      @param v2 2nd column vector for the new matrix
      @param v3 3rd column vector for the new matrix
      @param v4 4th column vector for the new matrix
   */
   public Matrix(Vector v1, Vector v2, Vector v3, Vector v4)
   {
      this.v1 = v1;  // Notice that we are not making
      this.v2 = v2;  // copies of the column vectors,
      this.v3 = v3;  // we are just making references
      this.v4 = v4;  // to them.
   }


   /**
      A scalar times a Matrix returns a (new) Matrix.

      @param s scalar value to multiply this matrix by
      @return new matrix object containing the scalar s times this matrix
   */
   public Matrix times(double s)
   {
      return new Matrix( this.v1.times(s), this.v2.times(s), this.v3.times(s), this.v4.times(s));
   }


   /**
      The product of two matrices returns a (new) Matrix.

      @param m matrix value to be multiplied by this matrix
      @return new matrix object containing this matrix times the matrix m
   */
   public Matrix times(Matrix m)
   {
      return new Matrix( this.times(m.v1), this.times(m.v2), this.times(m.v3), this.times(m.v4) );
   }


   /**
      A Matrix times a Vector returns a (new) Vector.

      @param v vector to be multiplied by this matrix
      @return new vector object containing this matrix times the vector v
   */
   public Vector times(Vector v)
   {
      return v1.times(v.x).plus(v2.times(v.y).plus(v3.times(v.z).plus(v4.times(v.w))));
   /*
      // Here is what this works out to be.
      Vector v1x = this.v1.times(v.x);
      Vector v2y = this.v2.times(v.y);
      Vector v3z = this.v3.times(v.z);
      Vector v4w = this.v4.times(v.w);
      Vector sum1 = v1x.plus(v2y);
      Vector sum2 = sum1.plus(v3z);
      Vector sum3 = sum2.plus(v4w);
      return sum3;
   */
   }


   /**
      A Matrix times a Vertex returns a (new) Vertex.

      @param v vertex to be multiplied by this matrix
      @return new vertex object containing this matrix times the vertex v
   */
   public Vertex times(Vertex v)
   {
      Vector sum = v1.times(v.x).plus(v2.times(v.y).plus(v3.times(v.z).plus(v4.times(v.w))));
      Vertex temp = new Vertex(sum.x, sum.y, sum.z, sum.w);
      // Copy color data from the original vertex to the new vertex.
      temp.setColor(v);
      return temp;
   }


   /**
      For debugging.

      @return String representation of this Matrix object
   */
   public String toString()
   {
      String result = "";
      int p = 5;      // the precision for the following format string
      int w = p + 4;  // the width for the following format string
      String format = "% "+w+"."+p+"f  % "+w+"."+p+"f  % "+w+"."+p+"f  % "+w+"."+p+"f";
      result += String.format("[[" + format + " ]\n",  v1.x, v2.x, v3.x, v4.x);
      result += String.format(" [" + format + " ]\n",  v1.y, v2.y, v3.y, v4.y);
      result += String.format(" [" + format + " ]\n",  v1.z, v2.z, v3.z, v4.z);
      result += String.format(" [" + format + " ]]\n", v1.w, v2.w, v3.w, v4.w);
    //result += String.format("[[% .5f  % .5f  % .5f  % .5f ]\n",  v1.x, v2.x, v3.x, v4.x);
    //result += String.format(" [% .5f  % .5f  % .5f  % .5f ]\n",  v1.y, v2.y, v3.y, v4.y);
    //result += String.format(" [% .5f  % .5f  % .5f  % .5f ]\n",  v1.z, v2.z, v3.z, v4.z);
    //result += String.format(" [% .5f  % .5f  % .5f  % .5f ]]\n", v1.w, v2.w, v3.w, v4.w);
      return result;
   }
}
