/*

*/
package scene;

import java.util.List;
import java.util.LinkedList;
import java.awt.Color;

/**
    A Model data structure represents a distinct geometric object in a Scene.
    A Model data structure is mainly a list of LineSegment objects. Each
    LineSegment object contains two Vertex objects with coordinates in the
    model's own model coordinate system.
<p>
    A Model object also contains a model matrix. The model matrix determines
    the model's location in the view coordinate system. The model matrix
    helps us solve the problem of placing (and moving) a model in a scene.
<p>
    When the renderer renders this model into the framebuffer, the first stage
    of the rendering pipeline multiplies every vertex in the model by the model's
    model matrix, which converts the coordinates in the vertex from the model's
    own model coordinate system to the camera's coordinate system (which is
    "shared" by all the other models in the scene).
<p>
    A Model also contains a List of sub Models. This list of sub Models helps
    us define what are called hierarchical, or nested, models. These are models
    that are made up of distinct parts and the distinct parts need to be able to
    both move individually and also move together as a unit. Since each sub Model
    is a Model object, each sub Model has its own model matrix (and maybe even
    more sub models). Each sub model's model matrix will position the sub model
    within its parent's model coordinate system. The "top level" model's matrix
    will place the whole model within the camera's coordinate system).
<p>
    As a simple, and common, example of a "nested model", think of a robot arm
    made up of a upper-arm, lower-arm, wrist, and five fingers. When you move
    the arm at its shoulder, all the arm's parts should move together. But when
    you move the arm at its elbow, only the lower-arm, wrist and fingers should
    move toghether. When you move the arm at its wrist, the wrist and fingers
    should move together. And each finger should be able to move by itself.
<p>
    The line segments represent the geometric object as a "wire-frame", that is,
    the geometric object is drawn as a collection of "edges". This is a fairly
    simplistic way of doing 3D graphics and we will improve this in later renderers.
<p>
   See
<br>    http://en.wikipedia.org/wiki/Wire-frame_model
   or
<br>    https://www.google.com/search?q=graphics+wireframe&tbm=isch
*/
public class Model
{
   public Matrix modelMatrix;

   public List<LineSegment> lineSegmentList = new LinkedList<LineSegment>();

   public List<Model> nestedModels = new LinkedList<Model>();


   /**
      Construct an empty model with the identity model matrix.
   */
   public Model()
   {
      modelMatrix = new Matrix(1.0); // identity matrix
   }

   /**
      Add a LineSegment (or LineSegments) to this model's list of line segments.

      @param lsArray array of LineSegments to add to this Model
   */
   public void addLineSegment(LineSegment... lsArray)
   {
      for (LineSegment ls : lsArray)
      {
         this.lineSegmentList.add(ls);
      }
   }

   /**
      Add a Model (or Models) to this model's list of sub-models.

      @param mArray array of sub-Models to add to this Model
   */
   public void addSubModel(Model... mArray)
   {
      for (Model m : mArray)
      {
         this.nestedModels.add(m);
      }
   }

   /**
      Get the index to a model from this model's list of sub-models.

      @param m Model to search for in this model's list of sub-models
      @return the index of the first occurrence of the specified model, or -1 if the model list list does not contain the model
   */
   public int indexOfSubModel(Model m)
   {
      return this.nestedModels.indexOf(m);
   }

   /**
      Get a reference to the model at the given index in this model's list of sub-models.

      @param index  index of the sub-Model to return
      @return  sub-Model at the specified position in the list of sub-Models
      @throws IndexOutOfBoundsException if the index is out of range (index < 0 || index >= size())
   */
   public Model getSubModel(int index)
   {
      return nestedModels.get(index);
   }

   /**
      Replace the model at the given index in this model's list of sub-models.

      @param  index index of the sub-Model to replace
      @param  m     Model to be stored at the specified position
      @return Model previously at the specified position
   */
   public Model setSubModel(int index, Model m)
   {
      return this.nestedModels.set(index, m);
   }

   /**
      Reset this model's model matrix to the identity matrix
   */
   public void model2Identity()
   {
      this.modelMatrix = new Matrix(1.0);
   }

   /**
      Multiply this model's current model matrix by the parameter m and make the result the new model matrix.

      @param m matrix to multiply this model's model matrix by
   */
   public void modelMult(Matrix m)
   {
      this.modelMatrix = this.modelMatrix.times(m);
   }

   /**
      Multiply this model's current model matrix by a rotation matrix and make the result the new model matrix.

      The rotation is by angle theta with axis of rotation given by (x, y, z).

      @param theta angle (in degrees) to rotate the model by
      @param x     x-coordinate of a vector on the axis of rotation
      @param y     y-coordinate of a vector on the axis of rotation
      @param z     z-coordinate of a vector on the axis of rotation
   */
   public void modelRotate(double theta, double x, double y, double z)
   {
      this.modelMult(new Matrix(theta, x, y, z));
   }

   /**
      Multiply this model's current model matrix by a scaling matrix and make the result the new model matrix.

      The scaling matrix scales by the amounts x, y and z along the x-axis, y-axis, and z-axis (respectively).

      @param x  scaling factor for the x-axis
      @param y  scaling factor for the y-axis
      @param z  scaling factor for the z-axis
   */
   public void modelScale(double x, double y, double z)
   {
      this.modelMult(new Matrix(x, y, z));
   }

   /**
      Multiply this model's current model matrix by a translation matrix and make the result the new model matrix.

      The translation matrix translates by the amounts x, y and z along the x-axis, y-axis, and z-axis (respectively).

      @param x  translation abount along the x-axis
      @param y  translation abount along the y-axis
      @param z  translation abount along the z-axis
   */
   public void modelTranslate(double x, double y, double z)
   {
      this.modelMult(new Matrix(new Vector(x, y, z)));
   }


   /**
      Set all the line segments in this model to the same color.

      @param c Color for all of this model's LineSegments
   */
   public void setColor(Color c)
   {
      for (LineSegment ls : this.lineSegmentList)
      {
         ls.setColor(c);
      }
   }

   /**
      Set all the line segments in this model to the same random color.
   */
   public void setColorRandom()
   {
      lineSegmentList.get(0).setColorRandom();
      Color c = lineSegmentList.get(0).v[0].getColor();
      for (LineSegment ls : this.lineSegmentList)
      {
         ls.setColor(c);
      }
   }

   /**
      Set all the line segments in this model to random colors.
   */
   public void setRandomColors()
   {
      for (LineSegment ls : this.lineSegmentList)
      {
         ls.setColorRandom();
      }
   }


   /**
      For debugging.

      @return String representation of this Model object
   */
   public String toString()
   {
      String result = "";
      result += "The Model matrix is\n";
      result += modelMatrix;
      result += "This Model has " + lineSegmentList.size() + " line segments\n";
      //result = "Printing out this Model's " + lineSegmentList.size() + " Line segments:\n";
      for (LineSegment ls : this.lineSegmentList)
      {
         result += ls.toString();
      }
      //result += "Done printing out Model\n";
      return result;
   }
}
