/*

*/
package scene;

/**
    A Vector object holds four doubles, which makes it a vector
    in 4-dimensional space.
<p>
    In computer graphics, we use 4-dimensional homogeneous coordinates
    to represent vectors (and points) in 3-dimensional space.
<p>
    Unlike a homogeneous point, a homogeneous vector usually has its
    fourth coordinate set to 0.
*/
public class Vector
{
   public double x, y, z, w; // a vector in homogenous coordinates


   /**
      Construct a default vector.
   */
   public Vector()
   {
      set(0.0, 0.0, 0.0);
   }


   /**
      Construct a new vector using the given x, y, and z coordinates.

      @param x x-coordinate of the new Vector
      @param y y-coordinate of the new Vector
      @param z z-coordinate of the new Vector
   */
   public Vector(double x, double y, double z)
   {
      set(x, y, z);
   }


   /**
      Construct a new vector that is a copy of another vector.

      @param v Vector to make a copy of
   */
   public Vector(Vector v)
   {
      set(v.x, v.y, v.z, v.w);
   }


   /**
      Construct a new vector using the given x, y, z, and w coordinates.

      This constructor is used to create the column vectors in a 4-by-4
      homogeneous matrix.

      @param x x-coordinate of the new Vector
      @param y y-coordinate of the new Vector
      @param z z-coordinate of the new Vector
      @param w w-coordinate of the new Vector
   */
   public Vector(double x, double y, double z, double w)
   {
      set(x, y, z, w);
   }


   /**
      Set the x, y, and z coordinates of this vector.

      @param x new x-coordinate for this Vector
      @param y new y-coordinate for this Vector
      @param z new z-coordinate for this Vector
   */
   public void set(double x, double y, double z)
   {
      set(x, y, z, 0.0);
   }


   /**
      Set the homogeneous coordinates of this vector.

      @param x new x-coordinate for this Vector
      @param y new y-coordinate for this Vector
      @param z new z-coordinate for this Vector
      @param w new w-coordinate for this Vector
   */
   public void set(double x, double y, double z, double w)
   {
      this.x = x;
      this.y = y;
      this.z = z;
      this.w = w;
   }


   /**
      A scalar times a Vector returns a (new) Vector.

      @param scalar number to multiply this Vector by
      @return a new Vector object that is the scalar times this Vector
   */
   public Vector times(double s)
   {
      return new Vector(s*x, s*y, s*z, s*w);
   }


   /**
      A Vector plus a Vector returns a (new) Vector.

      @param v Vector to add to this Vector
      @return a new Vector object that is the sum of this Vector and v
   */
   public Vector plus(Vector v)
   {
      return new Vector( x+(v.x), y+(v.y), z+(v.z), w+(v.w) );
   }


   /**
      A Vector minus a Vector returns a (new) Vector.

      @param v Vector to subtract from this Vector
      @return a new Vector object that is this Vector minus v
   */
   public Vector minus(Vector v)
   {
      return new Vector( x-(v.x), y-(v.y), z-(v.z), w-(v.w) );
   }


   /**
      The cross-product of two Vectors returns a (new) Vector.

      @param v Vector to multiply with this Vector
      @return a new Vector object that is the cross-product of this Vector and v
   */
   public Vector crossProduct(Vector v)
   {
      return new Vector((y*v.z)-(z*v.y), (z*v.x)-(x*v.z), (x*v.y)-(y*v.x));
   }


   /**
      The dot-product of two Vectors returns a scalar.

      @param v Vector to multiply with this Vector
      @return a double that is the dot-product of this Vector and v
   */
   public double dotProduct(Vector v)
   {
      return x*v.x + y*v.y + z*v.z;
   }


   /**
      Return the normalized version of this Vector.

      That is, return the Vector with length 1 that
      points in the same direction as this Vector.

      @return a new Vector that has length one and the same direction as this Vector
   */
   public Vector normalize()
   {
      double norm = Math.sqrt( x*x + y*y + z*z );
      return new Vector(x/norm, y/norm, z/norm);
   }


   /**
      For debugging.

      @return String representation of this Vector object
   */
   public String toString()
   {
      // Here is one way to get programmable precision and width.
      int p = 5;     // the precision for the following format string
      int t = p + 4; // the width for the following format string
      String format =      "[x,y,z,w] = [% "+t+"."+p+"f  % "+t+"."+p+"f  % "+t+"."+p+"f  % "+t+"."+p+"f]\n";
      return String.format( format, x, y, z, w);
    //return String.format("[x,y,z,w] = [% .5f  % .5f  % .5f  % .5f]", x, y, z, w);
   }
}
