/*

*/
package scene;

import java.util.Random;
import java.awt.Color;

/**
    A Vertex object has four doubles which represent the homogeneous
    coordinates of a point in 3-dimensional space. The fourth, homogeneous,
    coordinate will usually be 1, but in some stages of the graphics
    rendering pipeline it will be some other (non-zero) number.
<p>
    When a Vertex object is created in a client program, before the
    Vertex object moves down the graphics rendering pipeline, the
    coordinates in the Vertex will be in "camera coordinates".
<p>
    As a Vertex object moves down the graphics rendering pipeline,
    the coordinates in the Vertex will be transformed from one
    coordinate system to another.
<p>
    A vertex object also has a color (or "shade"). The color of a
    line segment is interpolated from the colors at its two endpoints.
*/
public class Vertex
{
   public double x, y, z, w;   // a vertex in homogenous coordinates
   public double r = 1.0, g = 1.0, b = 1.0;  // color at this vertex


   /**
      Construct a default vertex.
   */
   public Vertex()
   {
      set(0.0, 0.0, 0.0, 1.0);
   }


   /**
      Construct a new vertex (with homogeneous coordinates)
      using the given x, y, and z coordinates.

      @param x x-coordinate of the new Vertex
      @param y y-coordinate of the new Vertex
      @param z z-coordinate of the new Vertex
   */
   public Vertex(double x, double y, double z)
   {
      set(x, y, z, 1.0);
   }


   /**
      Construct a new vertex with the given homogeneous coordinates.

      @param x x-coordinate of the new Vertex
      @param y y-coordinate of the new Vertex
      @param z z-coordinate of the new Vertex
      @param w w-coordinate of the new Vertex
   */
   public Vertex(double x, double y, double z, double w)
   {
      set(x, y, z, w);
   }


   /**
      Construct a new vertex (with homogeneous coordinates)
      using the given x, y, and z coordinates and the given color.

      @param x x-coordinate of the new Vertex
      @param y y-coordinate of the new Vertex
      @param z z-coordinate of the new Vertex
      @param c Color of the new Vertex
   */
   public Vertex(double x, double y, double z, Color c)
   {
      set(x, y, z, 1.0);
      setColor(c);
   }


   /**
      Construct a new vertex with the given homogeneous coordinates
      and the given color.

      @param x x-coordinate of the new Vertex
      @param y y-coordinate of the new Vertex
      @param z z-coordinate of the new Vertex
      @param w w-coordinate of the new Vertex
      @param c Color of the new Vertex
   */
   public Vertex(double x, double y, double z, double w, Color c)
   {
      set(x, y, z, w);
      setColor(c);
   }


   /**
      Construct a new vertex that is a copy of another vertex.

      @param v Vertex to make a copy of
   */
   public Vertex(Vertex v)
   {
      set(v.x, v.y, v.z, v.w);
      setColor(v);
   }


   /**
      Set the coordinates of this vertex.

      @param x new x-coordinate for this Vertex
      @param y new y-coordinate for this Vertex
      @param z new z-coordinate for this Vertex
      @param w new w-coordinate for this Vertex
   */
   public void set(double x, double y, double z, double w)
   {
      this.x = x;
      this.y = y;
      this.z = z;
      this.w = w;
   }


   /**
      Matrix times Vertex mutates this Vertex.
      Needed for the transformation stages of the pipeline.

      @param matrix Matrix to multiply this Vertex by
   */
   public void times(Matrix matrix)
   {
      Vector col1 = matrix.v1;
      Vector col2 = matrix.v2;
      Vector col3 = matrix.v3;
      Vector col4 = matrix.v4;

      // dot product of the first matrix row with this vertex
      double x2 = (col1.x * this.x) + (col2.x * this.y)  + (col3.x * this.z) + (col4.x * this.w);

      // dot product of the second matrix row with this vertex
      double y2 = (col1.y * this.x) + (col2.y * this.y)  + (col3.y * this.z) + (col4.y * this.w);

      // dot product of the third matrix row with this vertex
      double z2 = (col1.z * this.x) + (col2.z * this.y)  + (col3.z * this.z) + (col4.z * this.w);

      // dot product of the fourth matrix row with this vertex
      double w2 = (col1.w * this.x) + (col2.w * this.y)  + (col3.w * this.z) + (col4.w * this.w);

      // Now mutate this Vertex object.
      this.x = x2;
      this.y = y2;
      this.z = z2;
      this.w = w2;
   }


   /**
      Needed for the perspective division step in the rendering pipeline.
   */
   public void perspectiveDivision()
   {
      double r = 1.0 / w;  // the reciprocal of the w-coordinate
      this.x *= r;
      this.y *= r;
      this.z *= r;
      this.w *= 1.0;
   }


   /**
      Get the color of this vertex.

      @return the color of this Vertex object
   */
   public Color getColor()
   {
      return new Color((float)this.r, (float)this.g, (float)this.b);
   }


   /**
      Set the color of this vertex using doubles between 0 and 1.

      @param r red color value for this Vertex as a double between 0 and 1
      @param g green color value for this Vertex as a double between 0 and 1
      @param b blue color value for this Vertex as a double between 0 and 1
   */
   public void setColor(double r, double g, double b)
   {
      this.r = r;
      this.g = g;
      this.b = b;

      if ( (r < 0.0) || (r > 1.0)
        || (g < 0.0) || (g > 1.0)
        || (b < 0.0) || (b > 1.0) )
      {
         System.err.println("ERROR! Invalid double color for vertex");
         System.err.println( this.toString() );
         System.err.printf("<r,g,b> = <% .5f  % .5f  % .5f>\n", r, g, b);
         System.exit(-1);
      }
   }


   /**
      Set the color of this vertex using ints between 0 and 255.

      @param r red color value for this Vertex as an integer between 0 and 255
      @param g green color value for this Vertex as an integer between 0 and 255
      @param b blue color value for this Vertex as an integer between 0 and 255
   */
   public void setColor(int r, int g, int b)
   {
      this.r = ((float)r)/(float)255;
      this.g = ((float)g)/(float)255;
      this.b = ((float)b)/(float)255;

      if ( (r < 0) || (r > 255)
        || (g < 0) || (g > 255)
        || (b < 0) || (b > 255))
      {
         System.err.println("ERROR! Invalid int color for vertex");
         System.err.println( this.toString() );
         System.err.printf("<r,g,b> = <%d  %d  %d>\n", r, g, b);
         System.exit(-1);
      }
   }


   /**
      Set the color of this vertex using a Color object.

      @param c Color for this Vertex object
   */
   public void setColor(Color c)
   {
      setColor(c.getRed(), c.getGreen(), c.getBlue());
   }


   /**
      Set the color of this vertex using the colors from another vertex.

      @param v Vertex object to get color values from
   */
   public void setColor(Vertex v)
   {
      // copy the color from v to this vertex
      setColor(v.r, v.g, v.b);
   }


   /**
      Set the color of this vertex to a random color.
   */
   public void setColorRandom()
   {
      Random generator = new Random();
      double r = generator.nextDouble();
      double g = generator.nextDouble();
      double b = generator.nextDouble();
      setColor(r, g, b);
   }


   /**
      For debugging.

      @return String representation of this Vertex object
   */
   public String toString()
   {
      // Here is one way to get programmable precision and width.
      int p = 5;     // the precision for the following format string
      int t = p + 4; // the width for the following format string
      String format =      "(x,y,z,w) = (% "+t+"."+p+"f  % "+t+"."+p+"f  % "+t+"."+p+"f  % "+t+"."+p+"f)\n";
      return String.format( format, x, y, z, w);
    //return String.format("(x,y,z,w) = (% .5f  % .5f  % .5f  % .5f)", x, y, z, w);
   }
}
