/*

*/
package scene.models;
import scene.*;

/**
   k number of triangles in each triangle fan at the top and bottom
   n number of rectangle strips between the top and bottom triangle fans
   r radius of the cylinder
   h height of the cylinder

   theta1 beginning longitude angle of the section
   theta2 ending longitude angle of the section
*/
public class CylinderSectionModel extends Model
{
   /**
      Three quarters of a cylinder.
   */
   public CylinderSectionModel( )
   {
      this(3, 2, 1, 1, 0, 1.5*Math.PI);
   }


   /**
      A full cylinder.
   */
   public CylinderSectionModel(int k, int n, double radius, double h)
   {
      this(k, n, radius, h, 0, 2*Math.PI);
   }


   public CylinderSectionModel(int k, int n, double radius, double h, double theta1, double theta2)
   {
      super();

      // Create the cylinder's geometry.

      // An array of vertices to be used to create triangles.
      Vertex[][] v = new Vertex[n+1][k+1];

      double deltaTheta = (theta2 - theta1)/k;

      // Create all the vertices along the cylinder wall.
      for (int j = 0; j <= k; j++)
      {
         double c = Math.cos(theta1 + j*deltaTheta);
         double s = Math.sin(theta1 + j*deltaTheta);
         for (int i = 0; i <= n; i++)
         {
            v[i][j] = new Vertex( radius*c, h-i*((2.0*h)/n), radius*s );
         }
      }
      // Vertex at the center of the top.
      Vertex topCenter = new Vertex(0.0, h, 0.0);
      // Vertex at the center of the bottom.
      Vertex bottomCenter = new Vertex(0.0, -h, 0.0);


      // Create all horizontal circles of latitude around the cylinder wall.
      LineSegment ls;
      for (int i = 0; i < n; i++)
      {
         for (int j = 0; j < k; j++)
         {
            ls = new LineSegment( new Vertex(v[i][j]), new Vertex(v[i][j+1]) );
            this.addLineSegment( ls );
         }
      }

      // Create the vertical lines from the top to the bottom.
      for (int j = 0; j <= k; j++)
      {
         ls = new LineSegment( new Vertex(v[0][j]), new Vertex(v[n][j]) );
         this.addLineSegment( ls );
      }

      // Create the triangle fan at the top.
      for (int j = 0; j <= k; j++)
      {
         ls = new LineSegment( new Vertex(topCenter), new Vertex(v[0][j]) );
         this.addLineSegment( ls );
      }

      // Create the triangle fan at the bottom.
      for (int j = 0; j <= k; j++)
      {
         ls = new LineSegment( new Vertex(bottomCenter), new Vertex(v[n][j]) );
         this.addLineSegment( ls );
      }
   }
}//CylinderSectionModel
