/*

*/
package scene.models;
import scene.*;

import java.util.Scanner;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.FileNotFoundException;
import java.util.regex.*;
import java.util.ArrayList;

/**
   <p>
   A simple demonstration of loading and drawing a basic OBJ file.
   </p>
   <p>
   A basic OBJ file is a text file that contains three kinds of lines: lines that begin with the character <code>'v'</code>, lines that begin with the character <code>'f'</code>, and lines that begin with the character <code>'#'</code>.
   </p>
   <p>
   A line in an OBJ file that begins with <code>'#'</code> is a comment line and can be ignored.
   </p>
   <p>
   A line in an OBJ file that begins with <code>'v'</code> is a line that describes a vertex in 3-dimensional space. The <code>'v'</code> will always be followed on the line by three doubles, the <code>x</code>, <code>y</code>, and <code>z</code> coordinates of the vertex.
   </p>
   <p>
   A line in an OBJ file that begins with <code>'f'</code> is a line that describes a "face", which for now will mean a triangle. The <code>'f'</code> will be followed on the line by three positive integers. The integers are the indices of the three vertices that make up the triangle. The "index" of a vertex is the order in which the vertex was listed in the OBJ file. So a line like this
   </p>
   <pre>
      f  2  4  1
   </pre>
   <p>
   would represent a triangle made up of the 2nd vertex read from the file, the 4th vertex read from the file, and the 1st vertex read from the file.
   </p>
*/
public class ObjSimpleModel extends Model
{
   public ObjSimpleModel(File modelFile)
   {
      super();

      // Open the OBJ file.
      String objName = null;
      FileInputStream fis = null;
      try
      {
         objName = modelFile.getCanonicalPath();
         fis = new FileInputStream( modelFile );
      }
      catch (FileNotFoundException e)
      {
         e.printStackTrace(System.err);
         System.err.printf("ERROR! Could not find OBJ file: %s\n", objName);
         System.exit(-1);
      }
      catch (IOException e)
      {
         e.printStackTrace(System.err);
         System.err.printf("ERROR! Could not open OBJ file: %s\n", objName);
         System.exit(-1);
      }

      // Get the geometry from the OBJ file.
      ArrayList<Vertex> verts = new ArrayList<Vertex>();
      try
      {
         // Pattern for parsing lines that start with "f"
         Pattern p = Pattern.compile("^(\\d*)[/]?(\\d*)[/]?(\\d*)");

         Scanner scanner = new Scanner(fis);
         while ( scanner.hasNext() )
         {
            String token = scanner.next();
            if ( token.startsWith("#")
              || token.startsWith("vt")
              || token.startsWith("vn")
              || token.startsWith("s")
              || token.startsWith("g")
              || token.startsWith("o")
              || token.startsWith("usemtl")
              || token.startsWith("mtllib") )
            {
               scanner.nextLine(); // skip over these lines
            }
            else if ( token.startsWith("v") )
            {
               double x = scanner.nextDouble();
               double y = scanner.nextDouble();
               double z = scanner.nextDouble();
               Vertex v = new Vertex(x, y, z);
               verts.add( v );
            }// parse vertex
            else if ( token.startsWith("f") )
            {
               LineSegment ls;
               // tokenize the rest of the line
               String restOfLine = scanner.nextLine();
               Scanner scanner2 = new Scanner( restOfLine );
               // parse three vertices and make two line segments
               int[] v = new int[3];
               for (int i = 0; i < 3; i++)
               {
                  // parse a "v/vt/vn" group
                  String faceGroup = scanner2.next();
                  Matcher m = p.matcher( faceGroup );
                  if ( m.find() )
                  {
                     v[i] = Integer.parseInt( m.group(1) );
                     String vt = m.group(2);  // don't need
                     String vn = m.group(3);  // don't need
                  }
                  else
                     System.err.println("Error: bad face: " + faceGroup);
               }
             //ls = new LineSegment(           verts.get(v[0]-1),             verts.get(v[1]-1) );
               ls = new LineSegment(new Vertex(verts.get(v[0]-1)), new Vertex(verts.get(v[1]-1)));
               this.addLineSegment( ls );
             //ls = new LineSegment(           verts.get(v[1]-1),             verts.get(v[2]-1) );
               ls = new LineSegment(new Vertex(verts.get(v[1]-1)), new Vertex(verts.get(v[2]-1)));
               this.addLineSegment( ls );

               // parse another vertex (if there is one) and make a line segment
               while (scanner2.hasNext())
               {
                  v[1] = v[2];
                  String faceGroup = scanner2.next();
                  Matcher m = p.matcher( faceGroup );
                  if ( m.find() )
                  {
                     v[2] = Integer.parseInt( m.group(1) );
                     String vt = m.group(2);  // don't need
                     String vn = m.group(3);  // don't need
                  }
                  else
                     System.err.println("Error: bad face: " + faceGroup);

                //ls = new LineSegment(           verts.get(v[1]-1),             verts.get(v[2]-1));
                  ls = new LineSegment(new Vertex(verts.get(v[1]-1)), new Vertex(verts.get(v[2]-1)));
                  this.addLineSegment( ls );
               }
               // close the line loop around this face
             //ls = new LineSegment(           verts.get(v[2]-1),             verts.get(v[0]-1));
               ls = new LineSegment(new Vertex(verts.get(v[2]-1)), new Vertex(verts.get(v[0]-1)));
               this.addLineSegment( ls );
            }// parse face
         }// parse one line
         fis.close();
      }
      catch (Exception e)
      {
         e.printStackTrace(System.err);
         System.err.printf("ERROR! Could not read OBJ file: %s\n", objName);
         System.exit(-1);
      }
   }
}//ObjSimpleModel
