/*

*/
package scene.models;
import scene.*;

/**
   This is a right, triangular prism with length 2h, side length 1, centered
   at the origin, and with its central axis on the y-axis.
   http://en.wikipedia.org/wiki/Prism_(geometry)

   Attached to each of the two triangular ends of the prism is a tetrahedron
   of height h2. So the total height is 2*(h + h2).
*/
public class PrismModel extends Model
{
   /**
      A prism of length 1 with a regular tetrahedron at the top and bottom.
   */
   public PrismModel( )
   {
      this(0, 1, (float)(Math.sqrt(2)/Math.sqrt(3)) );
   }


   /**
      A prism of length h with a regular tetrahedron at the top and bottom.
   */
   public PrismModel(double h)
   {
      this(0, h, (float)(Math.sqrt(2)/Math.sqrt(3)) );
   }


   /**
      A prism of length h with a regular tetrahedron at the top and bottom
      and n "lines of latitude" around the body of the prism.
   */
   public PrismModel(int n, double h)
   {
      this(n, h, (float)(Math.sqrt(2)/Math.sqrt(3)) );
   }


   /**
      A prism of length h with a tetrahedron at the top and bottom,
      where the face-edge-face angle of the tetrahedron is theta
      (with theta in radians!) and n "lines of latitude" around
      the body of the prism.

      If theta = 0, then there are no tetrahedrons at the ends of the prism.
      If theta = arctan(2*sqrt(2)), then the tetrahedrons are regular.
   */
   public PrismModel(int n, double h, double theta)
   {
      this(n, h, (float)(Math.tan(theta)/(2*Math.sqrt(3))) );
   }


   /**
      A prism of length h with a tetrahedron of height h2 at the top and bottom
      and n "lines of latitude" around the body of the prism.

      NOTE: h2 is a float so that we can distinguish this (private) constructor
            from the (public) previous one.
   */
   private PrismModel(int n, double h, float h2)
   {
      super();

      // Create the prism's geometry.
      double sqrt3 = Math.sqrt(3.0);
      double sqrt2 = Math.sqrt(2.0);
      Vertex[] v = new Vertex[8];

      v[0] = new Vertex( 1.0/sqrt3,  h,  0);    // three vertices around the top
      v[1] = new Vertex(-0.5/sqrt3,  h,  0.5);
      v[2] = new Vertex(-0.5/sqrt3,  h, -0.5);

      v[3] = new Vertex( 1.0/sqrt3, -h,  0);    // three vertices around the bottom
      v[4] = new Vertex(-0.5/sqrt3, -h,  0.5);
      v[5] = new Vertex(-0.5/sqrt3, -h, -0.5);

      v[6] = new Vertex( 0,  h+h2, 0);  // vertex at the top
      v[7] = new Vertex( 0, -h-h2, 0);  // vertex at the bottom

      // Create 15 line segments.
      LineSegment[] ls = new LineSegment[15];
      int i = 0;
      // 3 top faces
      ls[i++] = new LineSegment(new Vertex(v[6]), new Vertex(v[0]));
      ls[i++] = new LineSegment(new Vertex(v[6]), new Vertex(v[1]));
      ls[i++] = new LineSegment(new Vertex(v[6]), new Vertex(v[2]));
      // the top edge
      ls[i++] = new LineSegment(new Vertex(v[0]), new Vertex(v[1]));
      ls[i++] = new LineSegment(new Vertex(v[1]), new Vertex(v[2]));
      ls[i++] = new LineSegment(new Vertex(v[2]), new Vertex(v[0]));
      // three vertical edges
      ls[i++] = new LineSegment(new Vertex(v[0]), new Vertex(v[3]));
      ls[i++] = new LineSegment(new Vertex(v[1]), new Vertex(v[4]));
      ls[i++] = new LineSegment(new Vertex(v[2]), new Vertex(v[5]));
      // the bottom edge
      ls[i++] = new LineSegment(new Vertex(v[3]), new Vertex(v[4]));
      ls[i++] = new LineSegment(new Vertex(v[4]), new Vertex(v[5]));
      ls[i++] = new LineSegment(new Vertex(v[5]), new Vertex(v[3]));
      // 3 bottom faces
      ls[i++] = new LineSegment(new Vertex(v[7]), new Vertex(v[3]));
      ls[i++] = new LineSegment(new Vertex(v[7]), new Vertex(v[4]));
      ls[i++] = new LineSegment(new Vertex(v[7]), new Vertex(v[5]));

      // Add the line segments to the model.
      for (i = 0; i < ls.length; i++)
      {
         this.addLineSegment( ls[i] );
      }

      if (n > 0)
      {
         // Create n lines of "latitude" around the prism.
         Vertex[][] v2 = new Vertex[n][3];

         double delta_y = 2.0*h/(n+1);

         for (int j = 0; j < n; j++)
         {
            double y = -h + (j+1) * delta_y;
            v2[j][0] = new Vertex( 1.0/sqrt3,  y,  0);
            v2[j][1] = new Vertex(-0.5/sqrt3,  y,  0.5);
            v2[j][2] = new Vertex(-0.5/sqrt3,  y, -0.5);

            this.addLineSegment( new LineSegment(new Vertex(v2[j][0]), new Vertex(v2[j][1])) );
            this.addLineSegment( new LineSegment(new Vertex(v2[j][1]), new Vertex(v2[j][2])) );
            this.addLineSegment( new LineSegment(new Vertex(v2[j][2]), new Vertex(v2[j][0])) );
         }
      }
   }
}