/*

*/
package scene.models;
import scene.*;

/**
   k number of triangles in each triangle fan at the top and the bottom
   n number of rectangle strips and fans between the top and bottom angles

   r radius of the sphere

   theta1 beginning longitude angle of the section
   theta2 ending longitude angle of the section
   phi1   beginning latitude angle of the section
   phi2   ending latitude angle of the section
*/
public class SphereSectionModel extends Model
{
   /**
      Three quarters of a sphere.
   */
   public SphereSectionModel( )
   {
	   this(6, 6, 1, 0, 1.5*Math.PI, 0, Math.PI);
   }


   /**
      A full sphere.
   */
   public SphereSectionModel(int k, int n, double radius)
   {
      this(k, n, radius, 0, 2*Math.PI, 0, Math.PI);
   }


   public SphereSectionModel(int k, int n, double radius,
                             double theta1, double theta2, double phi1, double phi2)
   {
      super();

      // Create sphere section's geometry.

      double deltaPhi = (phi2 - phi1)/n;
      double deltaTheta = (theta2 - theta1)/k;

      double phiStart = phi1;
      double phiEnd   = phi2;
      if ( 0 == phi1 )
      {
         phiStart = deltaPhi;
         n = n - 1;
      }
      if ( Math.PI == phi2 )
      {
         phiEnd = Math.PI - deltaPhi;
         n = n-1;
      }

      // An array of vertices to be used to create  line segments.
      Vertex[][] v = new Vertex[n+1][k+1];

      // Create all the vertices.
      for (int j = 0; j <= k; j++)
      {
         double c1 = Math.cos(theta1 + j*deltaTheta);
         double s1 = Math.sin(theta1 + j*deltaTheta);
         for (int i = 0; i <= n; i++)
         {
            double c2 = Math.cos(phiStart + i*deltaPhi);
            double s2 = Math.sin(phiStart + i*deltaPhi);
            v[i][j] = new Vertex(radius*s2*c1, radius*c2, radius*s2*s1);
         }
      }
      // Vertex at the center of the top.
      Vertex topPole = new Vertex(0.0,  radius, 0.0);
      // Vertex at the center of the bottom.
      Vertex bottomPole = new Vertex(0.0, -radius, 0.0);

      // Create the horizontal circles of latitude around the sphere.
      LineSegment ls;
      for (int i = 0; i <= n; i++)
      {
         for (int j = 0; j < k; j++)
         {
            ls = new LineSegment( new Vertex(v[i][j]), new Vertex(v[i][j+1]) );
            this.addLineSegment( ls );
         }
      }

      // Create the vertical circles of longitude from the top to the bottom.
      for (int j = 0; j <= k; j++)
      {
         for (int i = 0; i < n; i++)
         {
            ls = new LineSegment( new Vertex(v[i][j]), new Vertex(v[i+1][j]) );
            this.addLineSegment( ls );
         }
      }

      // Create the triangle fan at the top (if needed).
      if ( 0 == phi1 )
      {
         for (int j = 0; j <= k; j++)
         {
            ls = new LineSegment( new Vertex(topPole), new Vertex(v[0][j]) );
            this.addLineSegment( ls );
         }
      }

      // Create the triangle fan at the bottom (if needed).
      if ( Math.PI == phi2 )
      {
         for (int j = 0; j <= k; j++)
         {
            ls = new LineSegment( new Vertex(bottomPole), new Vertex(v[n][j]) );
            this.addLineSegment( ls );
         }
      }
   }
}//SphereSectionModel
