/*

*/

import renderer.scene.*;
import renderer.models_TP.*; // models defined using triangle primitives
import renderer.pipeline.*;
import renderer.framebuffer.*;
import renderer.gui.*;

import java.io.File;
import java.awt.Color;
import java.awt.event.*;
import java.util.ArrayList;

/**

*/
@SuppressWarnings("serial")
public class InteractiveModels_R18 extends InteractiveFrame
{
   private double fovy = 90.0;
   private boolean changePerspectiveVolumeWithNear = true;

   private boolean render_z_buffer = false;

   private boolean doBackFaceCulling = true;
   private boolean frontFacingIsCCW = true;
   private boolean facesHaveTwoSides = true;
   private Color   backFaceColor = new Color(150, 200, 150); // light green

   private double eyeX = 0.0;
   private double eyeY = 0.0;
   private double eyeZ = 2.0;
   private double eyeRotX = 0.0;
   private double eyeRotY = 0.0;
   private double eyeRotZ = 0.0;

   private boolean letterbox = false;
   private double aspectRatio = 1.0;
   private double far    =  2.0;
   private double near   =  1.0;
   private double left   = -1.0;
   private double right  =  1.0;
   private double bottom = -1.0;
   private double top    =  1.0;
   private boolean showCamera = false;

   private boolean showMatrix = false;
   private double[] xTranslation = new double[10];
   private double[] yTranslation = new double[10];
   private double[] zTranslation = new double[10];
   private double[] xRotation = new double[10];
   private double[] yRotation = new double[10];
   private double[] zRotation = new double[10];
   private double[] scale = {1,1,1,1,1,1,1,1,1,1};

   private Scene scene;
   private ArrayList<Model> modelArray = new ArrayList<>();
   private int currentModel = 1;

   /**
      This constructor instantiates the Scene object
      and initializes it with appropriate geometry.
   */
   public InteractiveModels_R18(String title, int fbWidth, int fbHeight)
   {
      super(title, fbWidth, fbHeight);

      // Create the Scene object that we shall render.
      scene = new Scene();

      // Add four (empty) Positions to the Scene.
      scene.addPosition(new Position(), new Position(),
                        new Position(), new Position());

      // Set up the camera's location and orientation.
      scene.camera.viewTranslate(eyeX, eyeY, eyeZ);

      // Create several Model objects.
      modelArray.add( new ObjSimpleModel(new File("assets/apple.obj")) );
      modelArray.add( new ObjSimpleModel(new File("assets/cow.obj")) );
      modelArray.add( new ObjSimpleModel(new File("assets/galleon.obj")) );
      modelArray.add( new ObjSimpleModel(new File("assets/teapot.obj")) );
      modelArray.add( new ObjSimpleModel(new File("assets/cessna.obj")) );
      modelArray.add( new Sphere(1.0, 30, 30) );
      modelArray.add( new Cylinder(0.5, 1.0, 20, 20) );
      modelArray.add( new Torus(0.75, 0.25, 25, 25) );
      modelArray.add( new Cube2(15, 15, 15) );
      modelArray.add( new ObjSimpleModel(new File("assets/small_rhombicosidodecahedron.obj")) );
      modelArray.add( new renderer.models_LP.PanelXY(-7, 7, -1, 3) );  // wall
      modelArray.add( new renderer.models_LP.PanelXZ(-7, 7, -3, 1) );  // floor
      modelArray.add( new renderer.models_F.ObjSimpleModel(new File("assets/cessna.obj")) );

      // Give each model a random color.
      for (Model m : modelArray)
      {
         m.setRandomColor();
      }
      // Give all orientable primitives a back face color.
      for (Model m : modelArray)
      {
         m.setBackFaceColor(backFaceColor);
      }

      // Add four models to the Scene.
      int airplane = modelArray.size() - 1;
      int floor    = modelArray.size() - 2;
      int wall     = modelArray.size() - 3;
      scene.positionList.get(0).model = modelArray.get(1); // cow
      scene.positionList.get(1).model = modelArray.get(wall);
      scene.positionList.get(2).model = modelArray.get(floor);
      scene.positionList.get(3).model = modelArray.get(airplane);

      setBackFaceCulling(scene.positionList.get(0), doBackFaceCulling);
      setFrontFacingIsCCW(scene.positionList.get(0), frontFacingIsCCW);
      setFacesHaveTwoSides(scene.positionList.get(0), facesHaveTwoSides);

      // Position the wall and floor.
      scene.positionList.get(1).translate(0, 0, -3); // wall
      scene.positionList.get(2).translate(0, -1, 0); // floor
      // Push the last Position off to the side.
      scene.positionList.get(3).translate(3, 0, 0);
   }


   // Implement part of the KeyListener interface.
   @Override public void keyPressed(KeyEvent e)
   {
      //System.out.println( e );

      int keyCode = e.getKeyCode();
      // Only handle the four arrow keys.
      if (KeyEvent.VK_UP == keyCode   || KeyEvent.VK_DOWN == keyCode
       || KeyEvent.VK_LEFT == keyCode || KeyEvent.VK_RIGHT == keyCode)
      {
         if (KeyEvent.VK_UP == keyCode && e.getModifiers() != 0 )
         {
            eyeRotX += 1;
         }
         else if (KeyEvent.VK_DOWN == keyCode && e.getModifiers() != 0 )
         {
            eyeRotX -= 1;
         }
         else if (KeyEvent.VK_LEFT == keyCode && e.getModifiers() != 0 )
         {
            eyeRotY += 1;
         }
         else if (KeyEvent.VK_RIGHT == keyCode && e.getModifiers() != 0 )
         {
            eyeRotY -= 1;
         }
         else if (KeyEvent.VK_UP == keyCode)
         {
            eyeY += 0.1;
         }
         else if (KeyEvent.VK_DOWN == keyCode)
         {
            eyeY -= 0.1;
         }
         else if (KeyEvent.VK_LEFT == keyCode)
         {
            eyeX -= 0.1;
         }
         else if (KeyEvent.VK_RIGHT == keyCode)
         {
            eyeX += 0.1;
         }

         // Set up the camera's location and orientation.
         scene.camera.view2Identity();
         scene.camera.viewTranslate(eyeX, eyeY, eyeZ);
         scene.camera.viewRotateX(eyeRotX);
         scene.camera.viewRotateY(eyeRotY);

         if (showCamera)
         {
            System.out.println("eyeRotX = " + eyeRotX
                           + ", eyeRotY = " + eyeRotY);
            System.out.print( scene.camera.viewMatrix );
         }

         // Render again.
         setupViewport();
      }
   }


   // Implement part of the KeyListener interface.
   @Override public void keyTyped(KeyEvent e)
   {
      //System.out.println( e );

      char c = e.getKeyChar();
      if ('h' == c)
      {
         print_help_message();
         return;
      }
      else if ('d' == c)
      {
         Pipeline.debug = ! Pipeline.debug;
       //Clip.debug = ! Clip.debug;
       //Rasterize.debug = ! Rasterize.debug;
      }
      else if ( Character.isDigit(c) )
      {
         currentModel = Integer.parseInt( Character.toString(c) );
         // Change the Model in the first Position to the selected Model.
         scene.positionList.get(0).model = modelArray.get(currentModel);
         setBackFaceCulling(scene.positionList.get(0), doBackFaceCulling);
         setFrontFacingIsCCW(scene.positionList.get(0), frontFacingIsCCW);
         setFacesHaveTwoSides(scene.positionList.get(0), facesHaveTwoSides);
      }
      else if (' ' == c)
      {
         render_z_buffer = ! render_z_buffer;
      }
      else if ('g' == c)
      {
         doBackFaceCulling = ! doBackFaceCulling;
         setBackFaceCulling(scene.positionList.get(0), doBackFaceCulling);
       //setBackFaceCulling(scene.positionList.get(3), doBackFaceCulling);
         System.out.print("Back face culling is ");
         System.out.println(doBackFaceCulling ? "on" : "off");
      }
      else if ('G' == c)
      {
         frontFacingIsCCW = ! frontFacingIsCCW;
         setFrontFacingIsCCW(scene.positionList.get(0), frontFacingIsCCW);
       //setFrontFacingIsCCW(scene.positionList.get(3), frontFacingIsCCW);
         System.out.print("Front face is ");
         System.out.println(frontFacingIsCCW ? "CCW" : "CW");
      }
      else if ('t' == c)
      {
         facesHaveTwoSides = ! facesHaveTwoSides;
         setFacesHaveTwoSides(scene.positionList.get(0), facesHaveTwoSides);
       //setFacesHaveTwoSides(scene.positionList.get(3), facesHaveTwoSides);
         System.out.print("Faces have two sides is ");
         System.out.println(facesHaveTwoSides ? "On" : "Off");
      }
      else if ('P' == c)
      {
         scene.positionList.get(0).model.toPointCloud(2);
      }
      else if ('f' == c || 'F' == c)
      {
         // Move the camera's far plane.
         if ('f' == c)
         {
            far -= 0.01;
         }
         else
         {
            far += 0.01;
         }
      }
      else if ('n' == c || 'N' == c)
      {
         // Move the camera's near plane.
         if ('n' == c)
         {
            near -= 0.01;
         }
         else
         {
            near += 0.01;
         }
         // use the near plane to set the camera's field-of-view
         if (scene.camera.perspective)
         {
            if ( changePerspectiveVolumeWithNear )
            {
               fovy = 2.0 * (180./Math.PI) * Math.atan(top/near);
            }
            else
            {
               top = near * Math.tan((Math.PI/180.0)*fovy/2.0);
            }
         }
         else  // orthographic projection
         {
            fovy = 2.0 * (180./Math.PI) * Math.atan(top/near);
         }
      }
      else if ('r' == c || 'R' == c)
      {
         // Change the aspect ratio of the camera's view rectangle.
         if ('r' == c)
         {
            aspectRatio -= 0.1;
         }
         else
         {
            aspectRatio += 0.1;
         }

         // Adjust right and left.
         // (Keep the vertical field-of-view fixed.)
         right =  top * aspectRatio;
         left  = -right;
      }
      else if ('o' == c || 'O' == c)
      {
         // Change left, right, bottom, and top.
         // (Keep the aspect ratio fixed.)
         if ('o' == c)
         {
            left   += 0.1 * aspectRatio;
            right  -= 0.1 * aspectRatio;
            bottom += 0.1;
            top    -= 0.1;
         }
         else
         {
            left   -= 0.1 * aspectRatio;
            right  += 0.1 * aspectRatio;
            bottom -= 0.1;
            top    += 0.1;
         }

         // Update the camera's vertical field-of-view.
         fovy = 2.0 * (180./Math.PI) * Math.atan(top/near);
      }
      else if ('D' == c)// Decouple perspective camera's view volume from near.
      {
         changePerspectiveVolumeWithNear = ! changePerspectiveVolumeWithNear;
         if ( changePerspectiveVolumeWithNear )
         {
            System.out.println("The perspective view volume will change with near plane.");
         }
         else
         {
            System.out.println("The perspective view volume will NOT change with near plane.");
         }
      }
      else if ('l' == c)
      {
         letterbox = ! letterbox;
      }
      else if ('c' == c)
      {
         // Change the solid random color of the model.
         scene.positionList.get(0).model.setRandomColor();
      }
      else if ('C' == c)
      {
         // Change each color in the model to a random color.
         scene.positionList.get(0).model.setRandomColors();
         scene.positionList.get(0).model.setBackFaceColor(backFaceColor);
      }
      else if ('e' == c)
      {
         // Change the solid random color of each edge of the model.
         scene.positionList.get(0).model.setRandomPrimitiveColors();
         scene.positionList.get(0).model.setBackFaceColor(backFaceColor);
      }
      else if ( 'e' == c && e.isAltDown() )
      {
         // Change the random color of each vertex of the model.
         scene.positionList.get(0).model.setRandomVertexColors();
         scene.positionList.get(0).model.setBackFaceColor(backFaceColor);
      }
      else if ('E' == c)
      {
         // Change the random color of each end of each edge of the model.
         scene.positionList.get(0).model.setRainbowPrimitiveColors();
         scene.positionList.get(0).model.setBackFaceColor(backFaceColor);
      }
      else if ('a' == c)
      {
         Pipeline.doAntialiasing = ! Pipeline.doAntialiasing;
      }
      else if ('p' == c)
      {
         scene.camera.perspective = ! scene.camera.perspective;
      }
      else if ('M' == c)
      {
         showCamera = ! showCamera;
      }
      else if ('m' == c)
      {
         showMatrix = ! showMatrix;
      }
      else if ('s' == c) // Scale the model 10% smaller.
      {
         scale[currentModel] /= 1.1;
      }
      else if ('S' == c) // Scale the model 10% larger.
      {
         scale[currentModel] *= 1.1;
      }
      else if ('x' == c)
      {
         xTranslation[currentModel] -= 0.1;
      }
      else if ('X' == c)
      {
         xTranslation[currentModel] += 0.1;
      }
      else if ('y' == c)
      {
         yTranslation[currentModel] -= 0.1;
      }
      else if ('Y' == c)
      {
         yTranslation[currentModel] += 0.1;
      }
      else if ('z' == c)
      {
         zTranslation[currentModel] -= 0.1;
      }
      else if ('Z' == c)
      {
         zTranslation[currentModel] += 0.1;
      }
      else if ('u' == c)
      {
         xRotation[currentModel] -= 2.0;
      }
      else if ('U' == c)
      {
         xRotation[currentModel] += 2.0;
      }
      else if ('v' == c)
      {
         yRotation[currentModel] -= 2.0;
      }
      else if ('V' == c)
      {
         yRotation[currentModel] += 2.0;
      }
      else if ('w' == c)
      {
         zRotation[currentModel] -= 2.0;
      }
      else if ('W' == c)
      {
         zRotation[currentModel] += 2.0;
      }

      // Set the model-to-world transformation matrix.
      // The order of the transformations is important!
      Position model_p = scene.positionList.get(0);
      model_p.matrix2Identity();
      // Move the model relative to its new position.
      model_p.translate(xTranslation[currentModel],
                        yTranslation[currentModel],
                        zTranslation[currentModel]);
      model_p.rotateX(xRotation[currentModel]);
      model_p.rotateY(yRotation[currentModel]);
      model_p.rotateZ(zRotation[currentModel]);
      model_p.scale(scale[currentModel]);

      // Set up the camera's view volume.
      if (scene.camera.perspective)
      {
         scene.camera.projPerspective(fovy, aspectRatio, near, far);
      }
      else
      {
         scene.camera.projOrtho(fovy, aspectRatio, near, far);
      }

      if (showCamera && ('M'==c
           ||'n'==c||'N'==c||'o'==c||'O'==c||'r'==c||'R'==c||'p'==c))
      {
         System.out.print( scene.camera );
      }

      if (showMatrix && ('m'==c
           ||'s'==c||'x'==c||'y'==c||'z'==c||'u'==c||'v'==c||'w'==c
           ||'S'==c||'X'==c||'Y'==c||'Z'==c||'U'==c||'V'==c||'W'==c))
      {
         System.out.println("xRot = " + xRotation[currentModel]
                        + ", yRot = " + yRotation[currentModel]
                        + ", zRot = " + zRotation[currentModel]);
         System.out.print( model_p.modelMatrix );
      }

      // Render again.
      setupViewport();
   }


   // Implement part of the ComponentListener interface.
   @Override public void componentResized(ComponentEvent e)
   {
      //System.out.println( e );

      // Get the new size of the FrameBufferPanel.
      int w = this.fbp.getWidth();
      int h = this.fbp.getHeight();

      // Create a new FrameBuffer that fits the new window size.
      FrameBuffer fb = new FrameBuffer(w, h);
      this.fbp.setFrameBuffer(fb);

      // Render again.
      setupViewport();
   }


   // Get in one place the code to set up the viewport.
   private void setupViewport()
   {
      // Render again.
      // Get the size of the FrameBuffer.
      FrameBuffer fb = this.fbp.getFrameBuffer();
      int w = fb.getWidthFB();
      int h = fb.getHeightFB();
      // Create a viewport with the correct aspect ratio.
      if ( letterbox )
      {
         if ( aspectRatio <= w/(double)h )
         {
            int width = (int)(h*aspectRatio);
            int xOffset = (w - width)/2;
            fb.setViewport(xOffset, 0, width, h);
         }
         else
         {
            int height = (int)(w/aspectRatio);
            int yOffset = (h - height)/2;
            fb.setViewport(0, yOffset, w, height);
         }
         fb.clearFB(Color.darkGray);
         fb.clearVP(Color.black);
      }
      else // the viewport is the whole framebuffer
      {
         fb.clearFB(Color.black);
         fb.setViewport();
      }
      Pipeline.render(scene, fb);
      if ( render_z_buffer )
      {
         FrameBuffer zBuf = this.fbp.getFrameBuffer().convertZB2FB();
         this.fbp.getFrameBuffer().setViewport(0, 0, zBuf);
      }
      fbp.update();
      repaint();
   }


   /**
      Create an instance of this class which has
      the affect of creating the GUI application.
   */
   public static void main(String[] args)
   {
      print_help_message();

      // Define initial dimensions for a FrameBuffer.
      int width  = 1024;
      int height = 1024;
      // Create an InteractiveFrame containing a FrameBuffer
      // with the given dimensions. NOTE: We need to call the
      // InteractiveModels_R18 constructor in the Java GUI Event
      // Dispatch Thread, otherwise we get a race condition
      // between the constructor (running in the main() thread)
      // and the very first ComponentEvent (running in the EDT).
      javax.swing.SwingUtilities.invokeLater(
         new Runnable() // an anonymous inner class constructor
         {
            public void run() // implement the Runnable interface
            {
               // call the constructor that builds the gui
               new InteractiveModels_R18("Renderer 18", width, height);
            }
         }
      );
   }//main()


   private static void print_help_message()
   {
      System.out.println("Use the 'd' key to toggle debugging information on and off.");
      System.out.println("Use the digit keys '0' - '9' to choose a model.");
      System.out.println("Use the 'p' key to toggle between parallel and orthographic projection.");
      System.out.println("Use the x/X, y/Y, z/Z, keys to translate the model along the x, y, z axes.");
      System.out.println("Use the u/U, v/V, w/W, keys to rotate the model around the x, y, z axes.");
      System.out.println("Use the s/S keys to scale the size of the model.");
      System.out.println("Use the 'c' key to change the random solid model color.");
      System.out.println("Use the 'C' key to randomly change model's colors.");
      System.out.println("Use the 'e' key to change the random solid primitive colors.");
      System.out.println("Use the 'E' key to change the random primitive colors.");
      System.out.println("Use the 'a' key to toggle antialiasing on and off.");
      System.out.println("Use the n/N keys to move the camera's near plane.");
      System.out.println("Use the f/F keys to move the camera's far plane.");
      System.out.println("Use the o/O keys to change the size of the camera's view rectangle.");
      System.out.println("Use the r/R keys to change the aspect ratio of the camera's view rectangle.");
      System.out.println("Use the 'D' key to decouple the perspective camera's view volume from near.");
      System.out.println("Use the 'l' key to toggle letterboxing on and off.");
      System.out.println("Use the arrow keys to move the camera location left/right and up/down.");
      System.out.println("Use CTRL arrow keys to rotate the camera left/right and up/down.");
      System.out.println("Use the 'm' key to toggle showing the Model transformation matrix.");
      System.out.println("Use the 'M' key to toggle showing the Camera's two matrices.");
      System.out.println("Use the 'g' key to toggle back face culling.");
      System.out.println("Use the 'G' key to toggle CW vs CCW.");
      System.out.println("Use the 't' key to toggle two sided faces.");
      System.out.println("Use the 'P' key to convert the current model to a point cloud.");
      System.out.println("Use the ' ' key to toggle between the image buffer and z-buffer.");
      System.out.println("Use the 'h' key to redisplay this help message.");
   }


   private static void setBackFaceCulling(Position p, boolean doBackFaceCulling)
   {
      if (null != p.model)
      {
         p.model.setBackFaceCulling(doBackFaceCulling);
      }
      for (Position p2 : p.nestedPositions)
      {
         setBackFaceCulling(p2, doBackFaceCulling);
      }
   }


   private static void setFrontFacingIsCCW(Position p, boolean frontFacingIsCCW)
   {
      if (null != p.model)
      {
         p.model.setFrontFacingIsCCW(frontFacingIsCCW);
      }
      for (Position p2 : p.nestedPositions)
      {
         setFrontFacingIsCCW(p2, frontFacingIsCCW);
      }
   }


   private static void setFacesHaveTwoSides(Position p, boolean facesHaveTwoSides)
   {
      if (null != p.model)
      {
         p.model.setFacesHaveTwoSides(facesHaveTwoSides);
      }
      for (Position p2 : p.nestedPositions)
      {
         setFacesHaveTwoSides(p2, facesHaveTwoSides);
      }
   }
}
