/*

*/

import renderer.scene.*;
import renderer.models_TP.*;
import renderer.pipeline.*;
import renderer.framebuffer.*;
import renderer.gui.*;

import java.io.File;
import java.awt.Color;
import java.awt.event.*;

/**
   Put the camera at the origin of world coordinates looking
   down the negative z-axis. This way, world coordinates and
   view coordinates coincide.
<p>
   Put a sphere of radius 1/4 at center (0, 0, -5/4) and put
   another sphere of radius 1/4 at center (0, 0, -7/4) so
   the second sphere is right behind and just touching the
   first sphere. Initialize the near plane with near = 1, so
   the near plane is just touching the front of the first
   sphere (recall that the near plane is z = -near in view
   coordinates). Initialize the far plane with far = 2, so
   the far plane is just touching the back of the second
   sphere (recall that the far plane is z = -far in view
   coordinates).
<p>
   The n/N keys will move the near plane and the f/F keys will
   move the far plane. Each tap of one of these keys moves the
   near or far plane by 0.01, so it takes 50 taps of the N-key
   to move the near plane through the whole first sphere (or 50
   taps of the f-key to move the far plane through the whole
   second sphere). Notice that at the 25'th tap of the N-key, the
   first sphere starts to shrink in radius because at that point
   the near plane has passed through more than half of the sphere
   (and at the 25'th tap of the f-key, the second sphere starts to
   shrink in radius because at that point the far plane has passed
   through more than half of the sphere).
<p>
   After 50 taps of the f-key, only the first sphere remains
   and the far plane is just touching it, so 50 more taps of
   the f-key will make the far plane pass through it also.
<p>
   Similarly, after restarting the program, after 50 taps of
   the N-key, only the second sphere remains and the near plane
   is just touching it, so 50 more taps of the N-key will make
   the near plane pass through it also.
<p>
   Be sure to use the D-key to turn on and off the fixing of
   the camera's field-of-view. Notice how this affects moving
   the near plane.
<p>
   At any time you can switch from perspective projection to
   orthographic projection and it should not change how much
   of the sphere has been clipped off.
*/
public class TestNearAndFarClippingPlanes
{
   private static double fovy = 90.0;
   private static boolean changePerspectiveVolumeWithNear = true;

   private static double far    =  2.0;
   private static double near   =  1.0;
   private static double left   = -1.0;
   private static double right  =  1.0;
   private static double bottom = -1.0;
   private static double top    =  1.0;

   private static double xTranslation = 0.0;
   private static double yTranslation = 0.0;
   private static double zTranslation = 0.0;
   private static double xRotation = 0.0;
   private static double yRotation = 0.0;
   private static double zRotation = 0.0;

   private static void print_help_message()
   {
      System.out.println("Use the 'd' key to toggle debugging information on and off.");
      System.out.println("Use the x/X, y/Y, z/Z, keys to translate the model along the x, y, z axes.");
      System.out.println("Use the u/U, v/V, w/W, keys to rotate the model around the x, y, z axes.");
      System.out.println("Use the 'c' key to change the solid model colors.");
      System.out.println("Use the 'p' key to toggle between parallel and orthographic projection.");
      System.out.println("Use the n/N keys to move the camera's near plane.");
      System.out.println("Use the f/F keys to move the camera's far plane.");
      System.out.println("Use the 'D' key to decouple the perspective camera's view volume from near.");
      System.out.println("Use the 'h' key to redisplay this help message.");
   }

   public static void main(String[] args)
   {
      print_help_message();

      // Create two sphere Models with radius 1/4.
      Position p1 = new Position(new Sphere(0.25, 10, 10));
      Position p2 = new Position(new Sphere(0.25, 50, 50));


      // Create the Scene object that we shall render
      Scene scene = new Scene();

      // Add models to the Scene.
      scene.addPosition(p1, p2);

      // Put the sphere m1 at the edge of the initial near
      // plane (z = -1 in view coordinates).
      p1.translate(0, 0, -1.25);
      xTranslation =  0.0;
      yTranslation =  0.0;
      zTranslation = -1.25;

      // Put the sphere m2 right behind m1
      p2.translate(0, 0, -1.75);

      // Give the models random colors.
      for (Position p : scene.positionList)
         p.model.setRandomColor();

      // Set up the camera's view volume.
      scene.camera.projPerspective(left, right, bottom, top, near, far);

      // Create a InteractiveFrame and a FrameBuffer to render our scene into.
      int width  = 1024;
      int height = 1024;

      @SuppressWarnings("serial")
      InteractiveFrame app = new InteractiveFrame("Interactive JFrame", width, height)
      {
         // Implement part of the KeyListener interface.
         @Override public void keyTyped(KeyEvent e)
         {
            //System.out.println( e );
            char c = e.getKeyChar();
            if ('h' == c)
            {
               print_help_message();
               return;
            }
            else if ('d' == c)
            {
               Pipeline.debug = ! Pipeline.debug;
             //Rasterize.debug = ! Rasterize.debug;
             //Clip.debug = ! Clip.debug;
            }
            else if ('f' == c || 'F' == c)
            {
               // Move the camera's far plane.
               if ('f' == c)
               {
                  far -= 0.01;
               }
               else
               {
                  far += 0.01;
               }
            }
            else if ('n' == c || 'N' == c)
            {
               // Move the camera's near plane.
               if ('n' == c)
                  near -= 0.01;
               else
                  near += 0.01;

               if (scene.camera.perspective)
               {
                  if ( changePerspectiveVolumeWithNear )
                  {
                     fovy = 2.0 * (180./Math.PI) * Math.atan(top/near);
                  }
                  else
                  {
                     top    = near * Math.tan((Math.PI/180.0)*fovy/2.0);
                     bottom = -top;
                     right  = top;
                     left   = -right;
                  }
               }
               else  // orthographic projection
               {
                  fovy = 2.0 * (180./Math.PI) * Math.atan(top/near);
               }
            }
            else if ('D' == c)
            {
               changePerspectiveVolumeWithNear = ! changePerspectiveVolumeWithNear;
               if ( changePerspectiveVolumeWithNear )
               {
                  System.out.println("The perspective view volume will change with near plane.");
               }
               else
               {
                  System.out.println("The perspective view volume will NOT change with near plane.");
               }
            }
            else if ('p' == c)
            {
               scene.camera.perspective = ! scene.camera.perspective;
            }
            else if ('c' == c)
            {
               // Change the solid random color of each model.
               for (Position p : scene.positionList)
                     p.model.setRandomColor();
            }
            else if ('x' == c)
            {
               xTranslation -= 0.1;
            }
            else if ('X' == c)
            {
               xTranslation += 0.1;
            }
            else if ('y' == c)
            {
               yTranslation -= 0.1;
            }
            else if ('Y' == c)
            {
               yTranslation += 0.1;
            }
            else if ('z' == c)
            {
               zTranslation -= 0.01;
            }
            else if ('Z' == c)
            {
               zTranslation += 0.01;
            }
            else if ('u' == c)
            {
               xRotation -= 2.0;
            }
            else if ('U' == c)
            {
               xRotation += 2.0;
            }
            else if ('v' == c)
            {
               yRotation -= 2.0;
            }
            else if ('V' == c)
            {
               yRotation += 2.0;
            }
            else if ('w' == c)
            {
               zRotation -= 2.0;
            }
            else if ('W' == c)
            {
               zRotation += 2.0;
            }

            // Set the model-to-world transformation matrix.
            // The order of the transformations is very important!
            Position position = scene.positionList.get(0);
            position.matrix2Identity();
            position.translate(xTranslation, yTranslation, zTranslation);
            position.rotateX(xRotation);
            position.rotateY(yRotation);
            position.rotateX(zRotation);

            // Set up the camera's view volume.
            if (scene.camera.perspective)
            {
               scene.camera.projPerspective(left, right, bottom, top, near, far);
            }
            else
            {
               scene.camera.projOrtho(left, right, bottom, top, near, far);
            }
            if ('n' == c || 'N' == c || 'f' == c || 'F' == c)
               System.out.print(scene.camera);

            // Render again.
            this.getFrameBuffer().clearFB(Color.black);
            Pipeline.render(scene, this.getFrameBuffer());
            fbp.update();
            repaint();
         }

         // Implement part of the ComponentListener interface.
         @Override public void componentResized(ComponentEvent e)
         {
            //System.out.println( e );

            // Get the new size of the FrameBufferPanel.
            int w = this.fbp.getWidth();
            int h = this.fbp.getHeight();

            // Create a new FrameBuffer that fits the new window size.
            FrameBuffer fb = new FrameBuffer(w, h);
            this.fbp.setFrameBuffer(fb);
            fb.clearFB(Color.black);
            Pipeline.render(scene, fb);
            fbp.update();
            repaint();
         }
      };
   }
}
