/*

*/

package renderer.models_F;
import  renderer.scene.*;
import  renderer.scene.primitives.*;

/**
   Create a wireframe model of a partial right circular cone with its
   base parallel to the xz-plane and its apex on the positive y-axis.
<p>
   By a partial cone we mean a cone over a circular sector of the
   cone's base and also cutting off the top part of the cone (the
   part between the apex and a circle of latitude) leaving a frustum
   of the (partial) cone.

   @see Cone
   @see ConeFrustum
*/
public class ConeSector extends Model
{
   /**
      Create half of a right circular cone with its base in the xz-plane,
      a base radius of 1, height 1, and apex on the positive y-axis.
   */
   public ConeSector( )
   {
      this(1, 1, Math.PI/2, 3*Math.PI/2, 15, 8);
   }


   /**
      Create a part of the cone with its base in the xz-plane,
      a base radius of {@code r}, height {@code h}, and apex
      on the y-axis.
   <p>
      If {@code theta1 > 0} or {@code theta2 < 2pi},then the partial
      cone is a cone over the circular sector from angle {@code theta1}
      to angle {@code theta2}. In other words, the (partial) circles of
      latitude in the model extend from angle {@code theta1} to angle
      {@code theta2}.
   <p>
      The last two parameters determine the number of lines of longitude
      and the number of (partial) circles of latitude in the model.
   <p>
      If there are {@code n} circles of latitude in the model (including
      the bottom edge), then each line of longitude will have {@code n}
      line segments. If there are {@code k} lines of longitude, then each
      (partial) circle of latitude will have {@code k-1} line segments.
   <p>
      There must be at least four lines of longitude and at least
      one circle of latitude.

      @param r       radius of the base in the xz-plane
      @param h       height of the apex on the y-axis
      @param theta1  beginning longitude angle of the sector
      @param theta2  ending longitude angle of the sector
      @param n       number of circles of latitude around the cone
      @param k       number of lines of longitude
   */
   public ConeSector(double r,
                     double h,
                     double theta1, double theta2,
                     int n, int k)
   {
      this(r, h, h, theta1, theta2, n+1, k);
   }


   /**
      Create a part of the cone with its base in the xz-plane,
      a base radius of {@code r}, height {@code  h}, and apex
      on the y-axis.
   <p>
      If {@code 0 < t < h}, then the partial cone is a frustum
      with its base in the xz-plane and the top of the frustum at
      {@code y = t}.
   <p>
      If {@code theta1 > 0} or {@code theta2 < 2pi},then the partial
      cone is a cone over the circular sector from angle {@code theta1}
      to angle {@code theta2}. In other words, the (partial) circles of
      latitude in the model extend from angle {@code theta1} to angle
      {@code theta2}.
   <p>
      The last two parameters determine the number of lines of longitude
      (not counting one edge of any removed sector) and the number of
      (partial) circles of latitude (not counting the top edge of the
      frustum) in the model.
   <p>
      If there are {@code n} circles of latitude in the model (including
      the bottom edge but not the top edge of the frustum), then each
      line of longitude will have {@code n+1} line segments. If there are
      {@code k} lines of longitude (not counting one edge of any removed
      sector), then each (partial) circle of latitude will have {@code k}
      line segments.
   <p>
      There must be at least four lines of longitude and at least
      two circles of latitude.

      @param r       radius of the base in the xz-plane
      @param h       height of the apex on the y-axis
      @param t       top of the frustum of the come
      @param theta1  beginning longitude angle of the sector
      @param theta2  ending longitude angle of the sector
      @param n       number of circles of latitude around the cone
      @param k       number of lines of longitude
   */
   public ConeSector(double r,
                     double h,
                     double t,
                     double theta1, double theta2,
                     int n, int k)
   {
      super();

      if (n < 2) n = 2;
      if (k < 4) k = 4;
      if (t > h) t = h;

      // Create the cone's geometry.

      double deltaH = t / (n - 1);
      double deltaTheta = (theta2 - theta1) / (k - 1);

      // An array of vertices to be used to create faces.
      Vertex[][] v = new Vertex[n][k];

      // Create all the vertices.
      for (int j = 0; j < k; ++j) // choose an angle of longitude
      {
         double c = Math.cos(theta1 + j * deltaTheta);
         double s = Math.sin(theta1 + j * deltaTheta);
         for (int i = 0; i < n; ++i) // choose a circle of latitude
         {
            double slantRadius = r * (1 - i * deltaH / h);
            v[i][j] = new Vertex( slantRadius * c,
                                  i * deltaH,
                                  slantRadius * s );
         }
      }
      Vertex apex         = new Vertex(0, t, 0);
      Vertex bottomCenter = new Vertex(0, 0, 0);

      // Add all of the vertices to this model.
      for (int i = 0; i < n; ++i)
      {
         for (int j = 0; j < k; ++j)
         {
            addVertex( v[i][j] );
         }
      }
      addVertex(bottomCenter);
      int bottomCenterIndex = n * k;
//    addVertex(apex);
//    int apexIndex = bottomCenterIndex + 1;

      // Create the triangle fan at the bottom.
      for (int j = 0; j < k - 1; ++j)
      {  //                                       v[0][j]  v[0][j+1]
         addPrimitive(new Face(bottomCenterIndex,    j,       j+1));
      }

      // Create all the square strips around the cone wall.
      for (int i = 0; i < n - 1; ++i)
      {
         for (int j = 0; j < k - 1; ++j)
         {
            if ((i == n-2) && (t == h) ) // prevent degenerate squares
            {  //                    v[i][j]    v[i+1][j+1]   v[i][j+1]
               addPrimitive(new Face((i*k)+j, ((i+1)*k)+j+1, (i*k)+j+1));
            }
            else
            {  //                    v[i][j]   v[i+1][j]    v[i+1][j+1]    v[i][j+1]
               addPrimitive(new Face((i*k)+j, ((i+1)*k)+j, ((i+1)*k)+j+1, (i*k)+j+1));
            }
         }
      }

//      // Create the triangle fan at the top.
//      for (int j = 0; j < k - 1; ++j)
//      {  //                                v[n-1][j+1]    v[n-1][j]
//         addPrimitive(new Face(apexIndex, ((n-1)*k)+j+1, ((n-1)*k)+j));
//      }
   }
}//ConeSector
