/*

*/

package renderer.models_F;
import  renderer.scene.*;
import  renderer.scene.primitives.*;

/**
   Create a wireframe model of a cube with its center
   at the origin, having edge length 2, and with its
   corners at {@code (1, 1, 1)}.
<p>
   This version of the cube model has each face of
   the cube cut up by an n by m grid of lines.
<p>
   Here is a picture showing how the cube's eight corners
   are labeled.
<pre>{@code
                  v[4]
                   +-----------------+ v[5]
                  /|                /|
                /  |              /  |
              /    |            /    |
            /      |          /      |
      v[7] +-----------------+ v[6]  |
           |       |         |       |               y
           |       |         |       |               |
           |       |         |       |               |
           |  v[0] +---------|-------+ v[1]          |
           |      /          |      /                |
           |    /            |    /                  +----. x
           |  /              |  /                   /
           |/                |/                    /
           +-----------------+                    /
          v[3]              v[2]                 z
}</pre>

   @see Cube
   @see Cube3
   @see Cube4
*/
public class Cube2 extends Model
{
   /**
      Create a cube with its center at the origin, having edge
      length 2, with its corners at {@code (1, 1, 1)}. and
      with two perpendicular grid lines going across the middle
      of each of the cube's faces.
   */
   public Cube2( )
   {
      this(1, 1, 1);
   }


   /**
      Create a cube with its center at the origin, having edge
      length 2, with its corners at {@code (1, 1, 1)}, and
      with each of the cube's faces containing the given number
      of grid lines parallel to the x, y, and z directions.

      @param xGrid  number of grid lines perpendicular to the x-axis
      @param yGrid  number of grid lines perpendicular to the y-axis
      @param zGrid  number of grid lines perpendicular to the z-axis
   */
   public Cube2(int xGrid, int yGrid, int zGrid)
   {
      super();

      if (xGrid < 0) xGrid = 0;
      if (yGrid < 0) yGrid = 0;
      if (zGrid < 0) zGrid = 0;

      double xStep = 2.0 / (1 + xGrid);
      double yStep = 2.0 / (1 + yGrid);
      double zStep = 2.0 / (1 + zGrid);

      // Create the top and bottom sides of the cube.
      // Arrays of indices to be used to create the faces.
      int[][] vT = new int[xGrid+2][zGrid+2];
      int[][] vB = new int[xGrid+2][zGrid+2];
      int vIndex = 0;
      // Create vertices for the top face.
      for (int i = 0; i < xGrid + 2; ++i)
      {
         for (int j = 0; j < zGrid + 2; ++j)
         {
            addVertex(new Vertex(-1 + i * xStep,  1, -1 + j * zStep));
            vT[i][j] = vIndex;
            vIndex += 1;
         }
      }
      // Create vertices for the bottom face.
      for (int i = 0; i < xGrid + 2; ++i)
      {
         for (int j = 0; j < zGrid + 2; ++j)
         {
            addVertex(new Vertex(-1 + i * xStep, -1, -1 + j * zStep));
            vB[i][j] = vIndex;
            vIndex += 1;
         }
      }

      // Create the top and bottom faces.
      for (int i = 0; i < xGrid + 1; ++i)
      {
         for (int j = 0; j < zGrid + 1; ++j)
         {
            addPrimitive(new Face(vT[i  ][j  ],
                                  vT[i  ][j+1],
                                  vT[i+1][j+1],
                                  vT[i+1][j  ]));
            addPrimitive(new Face(vB[i  ][j  ],
                                  vB[i+1][j  ],
                                  vB[i+1][j+1],
                                  vB[i  ][j+1]));
         }
      }


      // Create the front and back sides of the cube.
      // Arrays of indices to be used to create the faces.
      int[][] vF = new int[yGrid+2][xGrid+2];
              vB = new int[yGrid+2][xGrid+2];
      // Create vertices for the front face.
      for (int i = 0; i < yGrid + 2; ++i)
      {
         for (int j = 0; j < xGrid + 2; ++j)
         {
            addVertex(new Vertex(-1 + j * xStep, -1 + i * yStep,  1));
            vF[i][j] = vIndex;
            vIndex += 1;
         }
      }
      // Create vertices for the back face.
      for (int i = 0; i < yGrid + 2; ++i)
      {
         for (int j = 0; j < xGrid + 2; ++j)
         {
            addVertex(new Vertex(-1 + j * xStep, -1 + i * yStep, -1));
            vB[i][j] = vIndex;
            vIndex += 1;
         }
      }
      // Create the faces.
      for (int i = 0; i < yGrid + 1; ++i)
      {
         for (int j = 0; j < xGrid + 1; ++j)
         {
            addPrimitive(new Face(vF[i  ][j  ],
                                  vF[i  ][j+1],
                                  vF[i+1][j+1],
                                  vF[i+1][j  ]));
            addPrimitive(new Face(vB[i  ][j  ],
                                  vB[i+1][j  ],
                                  vB[i+1][j+1],
                                  vB[i  ][j+1]));
         }
      }


      // Create the right and left sides of the cube.
      // Arrays of indices to be used to create the faces.
      int[][] vR = new int[yGrid+2][zGrid+2];
      int[][] vL = new int[yGrid+2][zGrid+2];
      // Create vertices for the right face.
      for (int i = 0; i < yGrid + 2; ++i)
      {
         for (int j = 0; j < zGrid + 2; ++j)
         {
            addVertex(new Vertex( 1, -1 + i * yStep, -1 + j * zStep));
            vR[i][j] = vIndex;
            vIndex += 1;
         }
      }
      // Create vertices for the left face.
      for (int i = 0; i < yGrid + 2; ++i)
      {
         for (int j = 0; j < zGrid + 2; ++j)
         {
            addVertex(new Vertex(-1, -1 + i * yStep, -1 + j * zStep));
            vL[i][j] = vIndex;
            vIndex += 1;
         }
      }
      // Create the faces.
      for (int i = 0; i < yGrid + 1; ++i)
      {
         for (int j = 0; j < zGrid + 1; ++j)
         {
            addPrimitive(new Face(vR[i  ][j  ],
                                  vR[i+1][j  ],
                                  vR[i+1][j+1],
                                  vR[i  ][j+1]));
            addPrimitive(new Face(vL[i  ][j  ],
                                  vL[i  ][j+1],
                                  vL[i+1][j+1],
                                  vL[i+1][j  ]));
         }
      }
   }
}//Cube2
