/*

*/

package renderer.models_F;
import  renderer.scene.*;
import  renderer.scene.primitives.*;

/**
   Create a wireframe model of a cube with its center
   at the origin, having edge length 2, and with its
   corners at {@code (1, 1, 1)}.
<p>
   This version of the cube model has each face of
   the cube cut up by a triangle fan.
<p>
   Here is a picture showing how the cube's eight corners
   are labeled.
<pre>{@code
                  v[4]
                   +-----------------+ v[5]
                  /|                /|
                /  |              /  |
              /    |            /    |
            /      |          /      |
      v[7] +-----------------+ v[6]  |
           |       |         |       |               y
           |       |         |       |               |
           |       |         |       |               |
           |  v[0] +---------|-------+ v[1]          |
           |      /          |      /                |
           |    /            |    /                  +----. x
           |  /              |  /                   /
           |/                |/                    /
           +-----------------+                    /
          v[3]              v[2]                 z
}</pre>

   @see Cube
   @see Cube2
   @see Cube4
*/
public class Cube3 extends Model
{
   /**
      Create a cube with its center at the origin, having edge
      length 2, with its corners at {@code (1, 1, 1)}. and
      with a triangle fan of four triangles in each face.
   */
   public Cube3( )
   {
      this(1, 1, 1);
   }


   /**
      Create a cube with its center at the origin, having edge
      length 2, with its corners at {@code (1, 1, 1)}, and
      with each of the cube's faces containing a triangle fan
      with the given number of triangles along each of the x,
      y, and z directions.
      <p>
      There must be at least one triangle along each direction.

      @param xCount  number of triangles along the x-direction
      @param yCount  number of triangles along the y-direction
      @param zCount  number of triangles along the z-direction
   */
   public Cube3(int xCount, int yCount, int zCount)
   {
      super();

      if (xCount < 1) xCount = 1;
      if (yCount < 1) yCount = 1;
      if (zCount < 1) zCount = 1;

      double xStep = 2.0 / xCount;
      double yStep = 2.0 / yCount;
      double zStep = 2.0 / zCount;

      addVertex(new Vertex(0,  0,  1)); // center front
      addVertex(new Vertex(0,  0, -1)); // center back
      addVertex(new Vertex(0,  1,  0)); // top center
      addVertex(new Vertex(0, -1,  0)); // center bottom
      addVertex(new Vertex( 1, 0,  0)); // center right
      addVertex(new Vertex(-1, 0,  0)); // center left

      int index = 0;
      int cFront  = index;
      int cBack   = index + 1;
      int cTop    = index + 2;
      int cBottom = index + 3;
      int rCenter = index + 4;
      int lCenter = index + 5;
      index += 6;

      // Triangles along all four edges parallel to the x-axis.
      double x = -1.0;
      addVertex(new Vertex(x,  1,  1));  // index - 4
      addVertex(new Vertex(x, -1,  1));  // index - 3
      addVertex(new Vertex(x,  1, -1));  // index - 2
      addVertex(new Vertex(x, -1, -1));  // index - 1
      index += 4;
      for (int i = 0; i < xCount; ++i)
      {
         x += xStep;
         addVertex(new Vertex(x,  1,  1));  // index + 0
         addVertex(new Vertex(x, -1,  1));  // index + 1
         addVertex(new Vertex(x,  1, -1));  // index + 2
         addVertex(new Vertex(x, -1, -1));  // index + 3
         // front face, top and bottom edges
         addPrimitive(new Face(index-4, cFront, index+0));
         addPrimitive(new Face(index-3, index+1, cFront));
         // back face, top and bottom edges
         addPrimitive(new Face(index-2, index+2, cBack));
         addPrimitive(new Face(index-1, cBack, index+3));
         // top face, front and back edges
         addPrimitive(new Face(index-4, index+0, cTop));
         addPrimitive(new Face(index-2, cTop, index+2));
         // bottom face, front and back edges
         addPrimitive(new Face(index-3, cBottom, index+1));
         addPrimitive(new Face(index-1, index+3, cBottom));
         index += 4;
      }

      // Triangles along all four edges parallel to the y-axis.
      double y = -1.0;
      addVertex(new Vertex( 1,  y,  1));  // index - 4
      addVertex(new Vertex(-1,  y,  1));  // index - 3
      addVertex(new Vertex( 1,  y, -1));  // index - 2
      addVertex(new Vertex(-1,  y, -1));  // index - 1
      index += 4;
      for (int i = 0; i < yCount; ++i)
      {
         y += yStep;
         addVertex(new Vertex( 1,  y,  1));  // index + 0
         addVertex(new Vertex(-1,  y,  1));  // index + 1
         addVertex(new Vertex( 1,  y, -1));  // index + 2
         addVertex(new Vertex(-1,  y, -1));  // index + 3
         // front face, right and left edges
         addPrimitive(new Face(index-4, index+0, cFront));
         addPrimitive(new Face(index-3, cFront,  index+1));
         // back face, right and left edges
         addPrimitive(new Face(index-2, cBack,   index+2));
         addPrimitive(new Face(index-1, index+3, cBack));
         // right face, front and back edges
         addPrimitive(new Face(index-4, rCenter, index+0));
         addPrimitive(new Face(index-2, index+2, rCenter));
         // left face, front and back edges
         addPrimitive(new Face(index-3, index+1, lCenter));
         addPrimitive(new Face(index-1, lCenter, index+3));
         index += 4;
      }

      // Triangles along all four edges parallel to the z-axis.
      double z = -1.0;
      addVertex(new Vertex( 1,  1,  z));  // index - 4
      addVertex(new Vertex(-1,  1,  z));  // index - 3
      addVertex(new Vertex( 1, -1,  z));  // index - 2
      addVertex(new Vertex(-1, -1,  z));  // index - 1
      index += 4;
      for (int i = 0; i < zCount; ++i)
      {
         z += zStep;
         addVertex(new Vertex( 1,  1,  z));  // index + 0
         addVertex(new Vertex(-1,  1,  z));  // index + 1
         addVertex(new Vertex( 1, -1,  z));  // index + 2
         addVertex(new Vertex(-1, -1,  z));  // index + 3
         // top face, right and left edges
         addPrimitive(new Face(index-4, cTop, index+0));
         addPrimitive(new Face(index-3, index+1, cTop));
         // bottom face, right and left edges
         addPrimitive(new Face(index-2, index+2, cBottom));
         addPrimitive(new Face(index-1, cBottom, index+3));
         // right face, top and bottom edges
         addPrimitive(new Face(index-4, index+0, rCenter));
         addPrimitive(new Face(index-2, rCenter, index+2));
         // left face, top and bottom edges
         addPrimitive(new Face(index-3, lCenter, index+1));
         addPrimitive(new Face(index-1, index+3, lCenter));
         index += 4;
      }
   }
}//Cube3
