/*

*/

package renderer.models_F;
import  renderer.scene.*;
import  renderer.scene.primitives.*;

/**
   Create a wireframe model of a sphere centered at the origin
   by recursively subdividing the faces of a tetrahedron.
   <p>
   Also use this subdivision process to draw Sierpinski triangles
   on the surface of the sphere.
*/
public class SphereSubdivided extends Model
{
   /**
      Create a sphere centered at the origin by recursively
      subdividing the faces of a tetrahedron four times.
   */
   public SphereSubdivided()
   {
      this(4);
   }


   /**
      Create a sphere centered at the origin by recursively
      subdividing the faces of a tetrahedron {@code n} times.

      @param n  number of recursive subdivisions
   */
   public SphereSubdivided(int n)
   {
      this(n, false, false);
   }


   /**
      Create a sphere centered at the origin by recursively
      subdividing the faces of a tetrahedron {@code n} times.
      <p>
      The {@code hole} parameter leaves out one of the original
      four triangle faces of the tetrahedron. This creates a hole
      in the final sphere that is useful for looking at the back
      side of the sphere.
      <p>
      The {@code sierpinski} parameter creates Sierpinski triangles
      on the sphere.

      @param n           number of recursive subdivisions
      @param hole        do not render one of the four triangles of the tetrahedron
      @param sierpinski  create Sierpinski triangles
   */
   public SphereSubdivided(int n, boolean hole, boolean sierpinski)
   {
      super();

      // Start with the tetrahedron's geometry.
      double sqr3inv = 1.0/Math.sqrt(3);
      Vertex v0 = new Vertex( sqr3inv,  sqr3inv,  sqr3inv);
      Vertex v1 = new Vertex(-sqr3inv,  sqr3inv, -sqr3inv);
      Vertex v2 = new Vertex( sqr3inv, -sqr3inv, -sqr3inv);
      Vertex v3 = new Vertex(-sqr3inv, -sqr3inv,  sqr3inv);

      // Subdivide each of the tetrahedron's four triangles.
      sub(n, v0, v1, v2, sierpinski);
      sub(n, v1, v3, v2, sierpinski);
      sub(n, v2, v3, v0, sierpinski);
      if (! hole) sub(n, v3, v1, v0, sierpinski);
   }


   /**
      Recursive helper function.

      @param n           number of recursive subdivisions
      @param v0          vertex of a triangle on the sphere
      @param v1          vertex of a triangle on the sphere
      @param v2          vertex of a triangle on the sphere
      @param sierpinski  create Sierpinski triangles
   */
   private void sub(int n, Vertex v0, Vertex v1, Vertex v2, boolean sierpinski)
   {
      if (0 == n)
      {
         int index = vertexList.size();
         addVertex(new Vertex(v0));  // prevent aliases in the vertex list
         addVertex(new Vertex(v1));
         addVertex(new Vertex(v2));
         addPrimitive(new Face(index+0, index+2, index+1)); // v0, v2, v1
      }
      else
      {
         // Subdivide each of the three edges.
         Vertex v3 = new Vertex(0.5*(v0.x + v1.x), 0.5*(v0.y + v1.y), 0.5*(v0.z + v1.z));
         Vertex v4 = new Vertex(0.5*(v1.x + v2.x), 0.5*(v1.y + v2.y), 0.5*(v1.z + v2.z));
         Vertex v5 = new Vertex(0.5*(v2.x + v0.x), 0.5*(v2.y + v0.y), 0.5*(v2.z + v0.z));
         // Normalize the subdivision points.
         double L3 = Math.sqrt(v3.x * v3.x + v3.y * v3.y + v3.z * v3.z);
         double L4 = Math.sqrt(v4.x * v4.x + v4.y * v4.y + v4.z * v4.z);
         double L5 = Math.sqrt(v5.x * v5.x + v5.y * v5.y + v5.z * v5.z);
         v3.set(v3.x / L3, v3.y / L3, v3.z / L3);
         v4.set(v4.x / L4, v4.y / L4, v4.z / L4);
         v5.set(v5.x / L5, v5.y / L5, v5.z / L5);
         // Recursively do another subdivision.
         sub(n-1, v0, v3, v5, sierpinski);
         sub(n-1, v5, v4, v2, sierpinski);
         sub(n-1, v3, v1, v4, sierpinski);
         if (! sierpinski) sub(n-1, v3, v4, v5, sierpinski);
      }
   }
}//SphereSubdivided
