/*

*/

package renderer.models_LP;
import  renderer.scene.*;
import  renderer.scene.primitives.*;

/**
   Create a wireframe model of a Menger Sponge centered at the origin.
<p>
   See <a href="https://en.wikipedia.org/wiki/Menger_sponge" target="_top">
                https://en.wikipedia.org/wiki/Menger_sponge</a>
*/
public class MengerSponge extends Model
{
   /**
      Create a Menger Sponge centered at the origin
      using three recursive iterations.
   */
   public MengerSponge()
   {
      this(3);
   }


   /**
      Create a Menger Sponge centered at the origin
      using {@code n} recursive iterations.

      @param n  number of recursive iterations
   */
   public MengerSponge(int n)
   {
      super();

      if (0 == n)
      {
         // Create a cube geometry.
         Vertex v0 = new Vertex(-1, -1, -1); // four vertices around the bottom face
         Vertex v1 = new Vertex( 1, -1, -1);
         Vertex v2 = new Vertex( 1, -1,  1);
         Vertex v3 = new Vertex(-1, -1,  1);
         Vertex v4 = new Vertex(-1,  1, -1); // four vertices around the top face
         Vertex v5 = new Vertex( 1,  1, -1);
         Vertex v6 = new Vertex( 1,  1,  1);
         Vertex v7 = new Vertex(-1,  1,  1);

         // Add the cube's vertices to the model.
         addVertex(v0, v1, v2, v3);
         addVertex(v4, v5, v6, v7);

         // Create 12 line segments using three primitives.
         addPrimitive(new LineLoop(0, 1, 2, 3));      // bottom face
         addPrimitive(new LineLoop(4, 5, 6, 7));      // top face
         addPrimitive(new Lines(0,4, 1,5, 2,6, 3,7)); // back and front faces
      }
      else
      {
         if (true) { // provide a way to erase the extra line segments
         // create 24 vertices for 12 line segements
         Vertex v0  = new Vertex(-1,       -1, -1.0/3.0); // eight vertices in the bottom face
         Vertex v1  = new Vertex(-1,       -1,  1.0/3.0);
         Vertex v2  = new Vertex( 1,       -1, -1.0/3.0);
         Vertex v3  = new Vertex( 1,       -1,  1.0/3.0);
         Vertex v4  = new Vertex(-1.0/3.0, -1, -1);
         Vertex v5  = new Vertex( 1.0/3.0, -1, -1);
         Vertex v6  = new Vertex(-1.0/3.0, -1,  1);
         Vertex v7  = new Vertex( 1.0/3.0, -1,  1);
         Vertex v8  = new Vertex(-1,        1, -1.0/3.0); // eight vertices in the top face
         Vertex v9  = new Vertex(-1,        1,  1.0/3.0);
         Vertex v10 = new Vertex( 1,        1, -1.0/3.0);
         Vertex v11 = new Vertex( 1,        1,  1.0/3.0);
         Vertex v12 = new Vertex(-1.0/3.0,  1, -1);
         Vertex v13 = new Vertex( 1.0/3.0,  1, -1);
         Vertex v14 = new Vertex(-1.0/3.0,  1,  1);
         Vertex v15 = new Vertex( 1.0/3.0,  1,  1);
         Vertex v16 = new Vertex(-1, -1.0/3.0, -1); // four vertices in the back face
         Vertex v17 = new Vertex(-1,  1.0/3.0, -1);
         Vertex v18 = new Vertex( 1, -1.0/3.0, -1);
         Vertex v19 = new Vertex( 1,  1.0/3.0, -1);
         Vertex v20 = new Vertex(-1, -1.0/3.0,  1); // four vertices in the front face
         Vertex v21 = new Vertex(-1,  1.0/3.0,  1);
         Vertex v22 = new Vertex( 1, -1.0/3.0,  1);
         Vertex v23 = new Vertex( 1,  1.0/3.0,  1);

         // Add these vertices to the model.
         addVertex(v0, v1, v2, v3, v4, v5, v6, v7);
         addVertex(v8, v9, v10, v11, v12, v13, v14, v15);
         addVertex(v16, v17, v18, v19);
         addVertex(v20, v21, v22, v23);

         // Create 12 line segments.
         addPrimitive(new LineSegment( 0,  1), // bottom face
                      new LineSegment( 2,  3),
                      new LineSegment( 4,  5),
                      new LineSegment( 6,  7),
                      new LineSegment( 8,  9), // top face
                      new LineSegment(10, 11),
                      new LineSegment(12, 13),
                      new LineSegment(14, 15),
                      new LineSegment(16, 17), // back face
                      new LineSegment(18, 19),
                      new LineSegment(20, 21), // front face
                      new LineSegment(22, 23));
         }
         nestedModels.add( subSponges(n - 1, -1, -1, -1) );
         nestedModels.add( subSponges(n - 1, -1, -1,  1) );
         nestedModels.add( subSponges(n - 1, -1,  1, -1) );
         nestedModels.add( subSponges(n - 1, -1,  1,  1) );
         nestedModels.add( subSponges(n - 1,  1, -1, -1) );
         nestedModels.add( subSponges(n - 1,  1, -1,  1) );
         nestedModels.add( subSponges(n - 1,  1,  1, -1) );
         nestedModels.add( subSponges(n - 1,  1,  1,  1) );
      }
   }


   /**
      Recursive helper function.
      <p>
      This function builds the eight sub models needed
      for one recusive step. These sub models will not
      be touching each other. In order to make the wireframe
      Menger Sponge look a bit better, we add line segments
      that connect some of the edges between the sub models.

      @param n    number of recursive iterations
      @param pmX  plus or minus 1 for x-direction
      @param pmY  plus or minus 1 for y-direction
      @param pmZ  plus or minus 1 for z-direction
      @return     {@link Model} holding sub tree of sponges
   */
   private Model subSponges(int n, int pmX, int pmY, int pmZ)
   {
      Model model = new Model();
      Matrix scale = Matrix.scale(1.0/3.0, 1.0/3.0, 1.0/3.0);
      Matrix translate = Matrix.translate(pmX*2.0/3.0, pmY*2.0/3.0, pmZ*2.0/3.0);
      model.nestedMatrix = translate.times(scale);
      if (0 == n) // stop the recursion
      {
         // Create a cube geometry.
         Vertex v0 = new Vertex(-1, -1, -1); // four vertices around the bottom face
         Vertex v1 = new Vertex( 1, -1, -1);
         Vertex v2 = new Vertex( 1, -1,  1);
         Vertex v3 = new Vertex(-1, -1,  1);
         Vertex v4 = new Vertex(-1,  1, -1); // four vertices around the top face
         Vertex v5 = new Vertex( 1,  1, -1);
         Vertex v6 = new Vertex( 1,  1,  1);
         Vertex v7 = new Vertex(-1,  1,  1);

         // Add the cube's vertices to the position's model.
         model.addVertex(v0, v1, v2, v3);
         model.addVertex(v4, v5, v6, v7);

         // Create 12 line segments using three primitives.
         model.addPrimitive(new LineLoop(0, 1, 2, 3));      // bottom face
         model.addPrimitive(new LineLoop(4, 5, 6, 7));      // top face
         model.addPrimitive(new Lines(0,4, 1,5, 2,6, 3,7)); // back and front faces
      }
      else
      {
         if (true) { // provide a way to erase the extra line segments
         // create 24 vertices for 12 line segements
         Vertex v0  = new Vertex(-1,       -1, -1.0/3.0); // eight vertices in the bottom face
         Vertex v1  = new Vertex(-1,       -1,  1.0/3.0);
         Vertex v2  = new Vertex( 1,       -1, -1.0/3.0);
         Vertex v3  = new Vertex( 1,       -1,  1.0/3.0);
         Vertex v4  = new Vertex(-1.0/3.0, -1, -1);
         Vertex v5  = new Vertex( 1.0/3.0, -1, -1);
         Vertex v6  = new Vertex(-1.0/3.0, -1,  1);
         Vertex v7  = new Vertex( 1.0/3.0, -1,  1);
         Vertex v8  = new Vertex(-1,        1, -1.0/3.0); // eight vertices in the top face
         Vertex v9  = new Vertex(-1,        1,  1.0/3.0);
         Vertex v10 = new Vertex( 1,        1, -1.0/3.0);
         Vertex v11 = new Vertex( 1,        1,  1.0/3.0);
         Vertex v12 = new Vertex(-1.0/3.0,  1, -1);
         Vertex v13 = new Vertex( 1.0/3.0,  1, -1);
         Vertex v14 = new Vertex(-1.0/3.0,  1,  1);
         Vertex v15 = new Vertex( 1.0/3.0,  1,  1);
         Vertex v16 = new Vertex(-1, -1.0/3.0, -1); // four vertices in the back face
         Vertex v17 = new Vertex(-1,  1.0/3.0, -1);
         Vertex v18 = new Vertex( 1, -1.0/3.0, -1);
         Vertex v19 = new Vertex( 1,  1.0/3.0, -1);
         Vertex v20 = new Vertex(-1, -1.0/3.0,  1); // four vertices in the front face
         Vertex v21 = new Vertex(-1,  1.0/3.0,  1);
         Vertex v22 = new Vertex( 1, -1.0/3.0,  1);
         Vertex v23 = new Vertex( 1,  1.0/3.0,  1);

         // Add these vertices to the position's model.
         model.addVertex(v0, v1, v2, v3, v4, v5, v6, v7);
         model.addVertex(v8, v9, v10, v11, v12, v13, v14, v15);
         model.addVertex(v16, v17, v18, v19);
         model.addVertex(v20, v21, v22, v23);

         // Create 12 line segments.
         model.addPrimitive(new LineSegment( 0,  1), // bottom face
                            new LineSegment( 2,  3),
                            new LineSegment( 4,  5),
                            new LineSegment( 6,  7),
                            new LineSegment( 8,  9), // top face
                            new LineSegment(10, 11),
                            new LineSegment(12, 13),
                            new LineSegment(14, 15),
                            new LineSegment(16, 17), // back face
                            new LineSegment(18, 19),
                            new LineSegment(20, 21), // front face
                            new LineSegment(22, 23));
         }
         model.nestedModels.add( subSponges(n - 1, -1, -1, -1) );
         model.nestedModels.add( subSponges(n - 1, -1, -1,  1) );
         model.nestedModels.add( subSponges(n - 1, -1,  1, -1) );
         model.nestedModels.add( subSponges(n - 1, -1,  1,  1) );
         model.nestedModels.add( subSponges(n - 1,  1, -1, -1) );
         model.nestedModels.add( subSponges(n - 1,  1, -1,  1) );
         model.nestedModels.add( subSponges(n - 1,  1,  1, -1) );
         model.nestedModels.add( subSponges(n - 1,  1,  1,  1) );
      }
      return model;
   }
}//MengerSponge
