/*

*/

package renderer.models_LP;
import  renderer.scene.*;
import  renderer.scene.primitives.*;

/**
   Create a wireframe model of a Sierpinski Triangle centered at the origin.
<p>
   See <a href="https://en.wikipedia.org/wiki/Sierpi%C5%84ski_triangle" target="_top">
                https://en.wikipedia.org/wiki/Sierpi%C5%84ski_triangle</a>
*/
public class SierpinskiTriangle extends Model
{
   /**
      Create a Sierpinski Triangle centered at the origin
      using five recursive iterations.
   */
   public SierpinskiTriangle()
   {
      this(7);
   }


   /**
      Create a Sierpinski Triangle centered at the origin
      using {@code n} recursive iterations.

      @param n  number of recursive iterations
   */
   public SierpinskiTriangle(int n)
   {
      super();

      if (0 == n)
      {
         // Create an elquilateral triangle.
         Vertex v0 = new Vertex( 1.0,  0,     0);
         Vertex v1 = new Vertex(-0.5,  0.866, 0);
         Vertex v2 = new Vertex(-0.5, -0.866, 0);

         // Add the triangles's vertices to the model.
         addVertex(v0, v1, v2);

         // Create a line loop.
         addPrimitive(new LineLoop(0, 1, 2));
      }
      else
      {
         nestedModels.add( subTriangles(n - 1,  0.5,   0) );
         nestedModels.add( subTriangles(n - 1, -0.25,  0.433) );
         nestedModels.add( subTriangles(n - 1, -0.25, -0.433) );
      }
   }


   /**
      Recursive helper function.
      <p>
      This function builds the three sub models needed
      for one recusive step.

      @param n   number of recursive iterations
      @param tX  translation in the x-direction
      @param tY  translation in the y-direction
      @return    {@link Model} holding sub tree of triangles
   */
   private Model subTriangles(int n, double tX, double tY)
   {
      Model model = new Model();
      Matrix scale = Matrix.scale(0.5, 0.5, 0.5);
      Matrix translate = Matrix.translate(tX, tY, 0);
      model.nestedMatrix = translate.times(scale);
      if (0 == n) // stop the recursion
      {
         // Create an elquilateral triangle.
         Vertex v0 = new Vertex( 1.0,  0,     0);
         Vertex v1 = new Vertex(-0.5,  0.866, 0);
         Vertex v2 = new Vertex(-0.5, -0.866, 0);

         // Add the triangle's vertices to the model.
         model.addVertex(v0, v1, v2);

         // Create a line loop.
         model.addPrimitive(new LineLoop(0, 1, 2));
      }
      else
      {
         model.nestedModels.add( subTriangles(n - 1,  0.5,   0) );
         model.nestedModels.add( subTriangles(n - 1, -0.25,  0.433) );
         model.nestedModels.add( subTriangles(n - 1, -0.25, -0.433) );
      }
      return model;
   }
}//SierpinskiTriangle
