/*

*/

package renderer.models_LP;
import  renderer.scene.*;
import  renderer.scene.primitives.*;

/**
   Create a wireframe model of a sphere centered at the origin.
<p>
   See <a href="https://en.wikipedia.org/wiki/Sphere" target="_top">
                https://en.wikipedia.org/wiki/Sphere</a>
<p>
   A sphere of radius {@code r} is the surface of revolution generated by
   revolving a half-circle in the xy-plane with radius {@code r} and center
   {@code (0,0,0)} around the y-axis.
<p>
   Here are parametric equations for the right half-circle in the xy-plane with
   radius {@code r} and center {@code (0,0,0)}, parameterized from the top down.
   <pre>{@code
      x(phi) = r * sin(phi)  \
      y(phi) = r * cos(phi)   |-  0 <= phi <= PI
      z(phi) = 0             /
   }</pre>
   Here is the 3D rotation matrix that rotates around the y-axis
   by {@code theta} radians, {@code 0 <= theta <= 2*PI}
   <pre>{@code
      [ cos(theta)   0   sin(theta)]
      [     0        1       0     ]
      [-sin(theta)   0   cos(theta)]
   }</pre>
   If we multiply the rotation matrix with the half-circle
   parameterization, we get a parameterization of the sphere.
   <pre>{@code
      [ cos(theta)   0   sin(theta)]   [r * sin(phi)]
      [     0        1       0     ] * [r * cos(phi)]
      [-sin(theta)   0   cos(theta)]   [     0      ]

      = ( r * sin(phi) * cos(theta).    \
          r * cos(phi),                  |- 0<=theta<=2*PI,  0<=phi<=PI
         -r * sin(phi) * sin(theta) )   /
   }</pre>
   See
     <a href="https://en.wikipedia.org/wiki/Sphere#Equations_in_three-dimensional_space" target="_top">
              https://en.wikipedia.org/wiki/Sphere#Equations_in_three-dimensional_space</a>

   @see SphereSector
*/
public class Sphere extends Model
{
   /**
      Create a sphere of radius 1 centered at the origin.
   */
   public Sphere( )
   {
      this(1, 15, 16);
   }


   /**
      Create a sphere of radius {@code r} centered at the origin.

      @param r  radius of the sphere
   */
   public Sphere(double r)
   {
      this(r, 15, 16);
   }


   /**
      Create a sphere of radius {@code r} centered at the origin.
   <p>
      The last two parameters determine the number of half circles
      of longitude and the number of circles of latitude in the model.
   <p>
      If there are {@code k} half circles of longitude, then each circle
      of latitude will have {@code k} line segments.
      If there are {@code n} circles of latitude, then each half circle
      of longitude will have {@code n+1} line segments.
   <p>
      There must be at least three half circles of longitude and
      at least one circle of latitude.

      @param r  radius of the sphere
      @param n  number of circles of latitude
      @param k  number of half circles of longitude
   */
   public Sphere(double r, int n, int k)
   {
      super();

      if (n < 1) n = 1;
      if (k < 3) k = 3;

      // Create the sphere's geometry.

      double deltaPhi = Math.PI / (n + 1);
      double deltaTheta = (2 * Math.PI) / k;

      // An array of vertices to be used to create line segments.
      Vertex[][] v = new Vertex[n][k];

      // Create all the vertices.
      for (int j = 0; j < k; ++j) // choose an angle of longitude
      {
         double c1 = Math.cos(j * deltaTheta);
         double s1 = Math.sin(j * deltaTheta);
         for (int i = 0; i < n; ++i) // choose an angle of latitude
         {
            double c2 = Math.cos(deltaPhi + i * deltaPhi);
            double s2 = Math.sin(deltaPhi + i * deltaPhi);
            v[i][j] = new Vertex( r * s2 * c1,
                                  r * c2,
                                 -r * s2 * s1 );
         }
      }
      Vertex northPole = new Vertex(0,  r, 0);
      Vertex southPole = new Vertex(0, -r, 0);

      // Add all of the vertices to this model.
      for (int i = 0; i < n; ++i)
      {
         for (int j = 0; j < k; ++j)
         {
            addVertex( v[i][j] );
         }
      }
      addVertex( northPole );
      addVertex( southPole );
      int northPoleIndex = n * k;
      int southPoleIndex = northPoleIndex + 1;

      // Create the horizontal circles of latitude around the sphere.
      for (int i = 0; i < n; ++i)
      {
         int[] indices = new int[k];
         for (int j = 0; j < k; ++j)
         {  //             v[i][j]
            indices[j] = (i * k) + j;
         }
         addPrimitive(new LineLoop(indices));
      }

      // Create the vertical lines of longitude.
      for (int j = 0; j < k; ++j)
      {
         int[] indices = new int[n];
         for (int i = 0; i < n; ++i)
         {  //             v[i][j]
            indices[i] = (i * k) + j;
         }
         addPrimitive(new LineStrip(indices));
      }


      // Create a line fan at the north pole.
      int[] indices = new int[k + 1];
      indices[0] = northPoleIndex;
      for (int j = 0; j < k; ++j)
      {  //                 v[0][j]
         indices[j + 1] = (0 * k) + j;
      }
      addPrimitive(new LineFan(indices));

      // Create a line fan at the south pole.
      indices = new int[k + 1];
      indices[0] = southPoleIndex;
      for (int j = 0; j < k; ++j)
      {  //                  v[n-1][j]
         indices[j + 1] = ((n-1) * k) + j;
      }
      addPrimitive(new LineFan(indices));
/*
      // Create the vertical half-circles of longitude from north to south pole.
      for (int j = 0; j < k; ++j)
      {  //                                v[0][j]
         addLineSegment( northPoleIndex, (0 * k) + j );

         for (int i = 0; i < n - 1; ++i)
         {  //                v[i][j]      v[i+1][j]
            addLineSegment( (i * k) + j, ((i+1) * k) + j );
         }
         //                 v[n-1][j]
         addLineSegment( ((n-1) * k) + j, southPoleIndex );
      }
*/
   }
}//Sphere
