/*

*/

package renderer.models_LP;
import  renderer.scene.*;
import  renderer.scene.primitives.*;

/**
   Create a wireframe model of a torus.
<p>
   See <a href="https://en.wikipedia.org/wiki/Torus" target="_top">
                https://en.wikipedia.org/wiki/Torus</a>
<p>
   This torus is the surface of revolution generated by revolving
   the circle in the xy-plane with radius {@code r2} and center
   {@code (r1,0,0)} around the y-axis. We are assuming that {@code r1 > r2}.
<p>
   Here are parametric equations for the circle in the xy-plane with
   radius {@code r2} and center {@code (r1,0,0)} and parameterized
   starting from the top, with parameter {@code 0 <= phi <= 2*PI}.
   <pre>{@code
      x(phi) = r1 + r2 * sin(phi)
      y(phi) =      r2 * cos(phi)
      z(phi) = 0
   }</pre>
   Here is the 3D rotation matrix that rotates around the y-axis
   by {@code theta} radians with {@code 0 <= theta <= 2*PI}.
   <pre>{@code
      [ cos(theta)   0   sin(theta)]
      [     0        1       0     ]
      [-sin(theta)   0   cos(theta)]
   }</pre>
   If we multiply the rotation matrix with the circle parameterization,
   we get a parameterization of the torus.
   <pre>{@code
      [ cos(theta)   0   sin(theta)]   [r1 + r2 * sin(phi)]
      [     0        1       0     ] * [     r2 * cos(phi)]
      [-sin(theta)   0   cos(theta)]   [        0         ]

      = ( r1*cos(theta) + r2*cos(theta)*sin(phi).
          r2*cos(phi),
         -r1*sin(theta) - r2*sin(theta)*sin(phi) )

      = ( (r1 + r2*sin(phi)) * cos(theta),
                r2*cos(phi),
         -(r1 + r2*sin(phi)) * sin(theta) )
   }</pre>
   See
     <a href="http://en.wikipedia.org/wiki/Torus#Geometry" target="_top">
              http://en.wikipedia.org/wiki/Torus#Geometry</a>

   @see TorusSector
*/
public class Torus extends Model
{
   /**
      Create a torus with a circle of revolution with radius 3/4
      and a cross section circle (circle of longitude)
      with radius 1/4.
   */
   public Torus( )
   {
      this(0.75, 0.25, 12, 16);
   }


   /**
      Create a torus with a circle of revolution with radius {@code r1}
      and a cross section circle (circle of longitude) with radius
      {@code r2}.

      @param r1  radius of the circle of revolution
      @param r2  radius of the cross section circle (circle of longitude)
   */
   public Torus(double r1, double r2)
   {
      this(r1, r2, 12, 16);
   }


   /**
      Create a torus with a circle of revolution with radius {@code r1}
      and a cross section circle (circle of longitude) with radius
      {@code r2}.
   <p>
      The last two parameters determine the number of circles of
      longitude and the number of circles of latitude in the model.
   <p>
      If there are {@code n} circles of latitude, then each circle
      of longitude will have {@code n} line segments.
      If there are {@code k} circles of longitude, then each circle
      of latitude will have {@code k} line segments.
   <p>
      There must be at least three circles of longitude and at least
      three circles of latitude.

      @param r1  radius of the circle of revolution
      @param r2  radius of the cross section circle (circle of longitude)
      @param n   number of circles of latitude
      @param k   number of circles of longitude
   */
   public Torus(double r1, double r2, int n, int k)
   {
      super();

      if (n < 3) n = 3;
      if (k < 3) k = 3;

      // Create the torus's geometry.

      double deltaPhi = (2 * Math.PI) / n;
      double deltaTheta = (2 * Math.PI) / k;

      // An array of vertices to be used to create line segments.
      Vertex[][] v = new Vertex[n][k];

      // Create all the vertices.
      for (int j = 0; j < k; ++j) // choose a rotation around the y-axis
      {
         double c1 = Math.cos(j * deltaTheta);
         double s1 = Math.sin(j * deltaTheta);
         for (int i = 0; i < n; ++i)  // go around a cross section circle
         {
            double c2 = Math.cos(i * deltaPhi);
            double s2 = Math.sin(i * deltaPhi);
            v[i][j] = new Vertex( (r1 + r2*s2) * c1,
                                        r2*c2,
                                 -(r1 + r2*s2) * s1 );
         }
      }

      // Add all of the vertices to this model.
      for (int i = 0; i < n; ++i)
      {
         for (int j = 0; j < k; ++j)
         {
            addVertex( v[i][j] );
         }
      }

      // Create the vertical cross-section circles.
      for (int j = 0; j < k; ++j) // choose a rotation around the y-axis
      {
         int[] indices = new int[n];
         for (int i = 0; i < n; ++i) // go around a cross section circle
         {  //             v[i][j]
            indices[i] = (i * k) + j;
         }
         addPrimitive(new LineLoop(indices));
      }

      // Create all the horizontal circles around the torus.
      for (int i = 0; i < n; ++i) //choose a rotation around the cross section
      {
         int[] indices = new int[k];
         for (int j = 0; j < k; ++j) // go around a horizontal circle
         {  //              v[i][j]
            indices[j] =  (i * k) + j;
         }
         addPrimitive(new LineLoop(indices));
      }
   }
}//Torus
