/*

*/

package renderer.models_T;
import  renderer.scene.*;
import  renderer.scene.primitives.*;

/**
   Create a solid model of a frustum of a right circular cone
   with its base in the xz-plane.
<p>
   See <a href="https://en.wikipedia.org/wiki/Frustum" target="_top">
                https://en.wikipedia.org/wiki/Frustum</a>

   @see Cone
   @see ConeSector
*/
public class ConeFrustum extends Model
{
   /**
      Create a frustum of a right circular cone with its base in the
      xz-plane, a base radius of 1, top radius of 1/2, and height 1/2.
   */
   public ConeFrustum( )
   {
      this(1.0, 0.5, 0.5, 7, 16);
   }


   /**
      Create a frustum of a right circular cone with its base in the
      xz-plane, a base radius of {@code r}, top of the frustum at
      height {@code h}, and with the cone's apex on the y-axis at
      height {@code a}.
   <p>
      There must be at least three lines of longitude and at least
      two circles of latitude.

      @param n  number of circles of latitude
      @param k  number of lines of longitude
      @param r  radius of the base in the xz-plane
      @param h  height of the frustum
      @param a  height of the apex of the cone
   */
   public ConeFrustum(int n, int k, double r, double h, double a)
   {
      this(r, (1 - h/a)*r, h, n, k);
   }


   /**
      Create a frustum of a right circular cone with its base in the
      xz-plane, a base radius of {@code r1}, top radius of {@code r2},
      and height {@code h}.
   <p>
      This model works with either {@code r1 > r2} or {@code r1 < r2}.
      In other words, the frustum can have its "apex" either above or
      below the xz-plane.
   <p>
      There must be at least three lines of longitude and at least
      two circles of latitude.

      @param r1  radius of the base of the frustum
      @param r2  radius of the top of the frustum
      @param h   height of the frustum
      @param n   number of circles of latitude
      @param k   number of lines of longitude
   */
   public ConeFrustum(double r1, double r2, double h, int n, int k)
   {
      super();

      if (n < 2) n = 2;
      if (k < 3) k = 3;

      // Create the frustum's geometry.

      double deltaTheta = (2 * Math.PI) / k;

      // An array of vertices to be used to create triangles.
      Vertex[][] v = new Vertex[n][k];

      // Create all the vertices.
      for (int j = 0; j < k; ++j) // choose an angle of longitude
      {
         double c = Math.cos(j * deltaTheta);
         double s = Math.sin(j * deltaTheta);
         for (int i = 0; i < n; ++i) // choose a circle of latitude
         {
            double slantRadius = (i/(double)(n-1)) * r1 + ((n-1-i)/(double)(n-1)) * r2;
            v[i][j] = new Vertex( slantRadius * c,
                                  h - (i*h)/(n-1),
                                  slantRadius * s );
         }
      }
      Vertex topCenter = new Vertex(0, h, 0);
      Vertex bottomCenter = new Vertex(0, 0, 0);

      // Add all of the vertices to this model.
      for (int i = 0; i < n; ++i)
      {
         for (int j = 0; j < k; ++j)
         {
            addVertex( v[i][j] );
         }
      }
      addVertex(topCenter,
                bottomCenter);
      int topCenterIndex = n * k;
      int bottomCenterIndex = topCenterIndex + 1;


      // Create all the triangle strips along the cone wall.
      for (int i = 0; i < n-1; ++i)
      {
         for (int j = 0; j < k-1; ++j)
         {  //                        v[i][j]  v[i][j+1]  v[i+1][j+1]
            addPrimitive(new Triangle((i*k)+j, (i*k)+j+1, ((i+1)*k)+j+1));
            //                        v[i][j]   v[i+1][j+1]    v[i+1][j]
            addPrimitive(new Triangle((i*k)+j, ((i+1)*k)+j+1, ((i+1)*k)+j));
         }
         //                        v[i][k-1]  v[i][0]   v[i+1][0]
         addPrimitive(new Triangle((i*k)+k-1, (i*k)+0, ((i+1)*k)+0));
         //                         v[i][k-1]   v[i+1][0]    v[i+1][k-1]
         addPrimitive(new Triangle((i*k)+k-1, ((i+1)*k)+0, ((i+1)*k)+k-1));
      }

      // Create the triangle fan at the top.
      for (int j = 0; j < k-1; ++j)
      {  //                                      v[0][j+1]  v[0][j]
         addPrimitive(new Triangle(topCenterIndex,   j+1,       j));
      }
      //                                       v[0][0]    v[0][k-1]
      addPrimitive(new Triangle(topCenterIndex,   0,        k-1));

      // Create the triangle fan at the bottom.
      for (int j = 0; j < k-1; ++j)
      {  //                                            v[n-1][j]    v[n-1][j+1]
         addPrimitive(new Triangle(bottomCenterIndex, ((n-1)*k)+j, ((n-1)*k)+j+1));
      }
      //                                            v[n-1][k-1]     v[n-1][0]
      addPrimitive(new Triangle(bottomCenterIndex, ((n-1)*k)+k-1, ((n-1)*k)+0));
   }
}//ConeFrustum
