/*

*/

package renderer.models_T;
import  renderer.scene.*;
import  renderer.scene.primitives.*;

/**
   Create a solid model of a right square pyramid with its
   base in the xz-plane and its apex on the positive y-axis.
<p>
   See <a href="https://en.wikipedia.org/wiki/Pyramid_(geometry)" target="_top">
                https://en.wikipedia.org/wiki/Pyramid_(geometry)</a>

   @see PyramidFrustum
*/
public class Pyramid extends Model
{
   /**
      Create a right square pyramid with its base in the xz-plane,
      a base side length of 2, height 1, and apex on the positive y-axis.
   */
   public Pyramid( )
   {
      this(2.0, 1.0, 15, 4);
   }


   /**
      Create a right square pyramid with its base in the xz-plane,
      a base length of {@code s}, height {@code h}, and apex on the
      positive y-axis.

      @param s  side length of the base in the xz-plane
      @param h  height of the apex on the y-axis
   */
   public Pyramid(double s, double h)
   {
      super();

      // Create the pyramid's geometry.
      Vertex v0 = new Vertex(-s/2, 0, -s/2);  // base
      Vertex v1 = new Vertex(-s/2, 0,  s/2);
      Vertex v2 = new Vertex( s/2, 0,  s/2);
      Vertex v3 = new Vertex( s/2, 0, -s/2);
      Vertex v4 = new Vertex(  0,  h,   0);   // apex
      addVertex(v0, v1, v2, v3, v4);

      // Create 6 triangles.
      // Make sure the triangles are all
      // oriented in the same way!
      addPrimitive(new Triangle(0, 1, 3),  // 2 base triangles
                   new Triangle(2, 3, 1),
                   new Triangle(4, 0, 1),  // 4 sides
                   new Triangle(4, 1, 2),
                   new Triangle(4, 2, 3),
                   new Triangle(4, 3, 0));
   }


   /**
      Create a right square pyramid with its base in the xz-plane,
      a base length of {@code s}, height {@code h}, and apex on the
      positive y-axis.

      @param s  side length of the base in the xz-plane
      @param h  height of the apex on the y-axis
      @param n  number of lines of latitude around the body of the pyramid
      @param k  number of triangles in the triangle fan at the top of each side
   */
   public Pyramid(double s, double h, int n, int k)
   {
      this(s, h, n, k, false);
   }


   /**
      Create a right square pyramid with its base in the xz-plane,
      a base length of {@code s}, height {@code h}, and apex on the
      positive y-axis.
   <p>
      The last parameter provides a choice between having a square
      grid of lines or a line fan in the base of the pyramid.

      @param s  side length of the base in the xz-plane
      @param h  height of the apex on the y-axis
      @param n  number of lines of latitude around the body of the pyramid
      @param k  number of triangles in the triangle fan at the top of each side
      @param grid  choose either a square grid or a line fan in the base
   */
   public Pyramid(double s, double h, int n, int k, boolean grid)
   {
      super();

      if (n < 1) n = 1;
      if (k < 1) k = 1;

      // Create the pyramid's geometry.
      double deltaH = h / n;

      // An array of vertices to be used to create triangles.
      Vertex[][] v = new Vertex[n][4*k + 1];

      // Create the vertices.
      for (int i = 0; i < n; ++i) // choose a height of latitude
      {
         double y = i * deltaH;
       //double slantSide = s * (1.0 - i*(1.0 / n));
         double slantSide = s - i*(s / n);
         double deltaS = slantSide / k;

         for (int j = 0; j < k; ++j)
         {
            v[i][j] = new Vertex(-slantSide/2 + j*deltaS,
                                  y,
                                 -slantSide/2);
         }
         for (int j = 0; j < k; ++j)
         {
            v[i][k+j] = new Vertex( slantSide/2,
                                    y,
                                   -slantSide/2 + j*deltaS);
         }
         for (int j = 0; j < k; ++j)
         {
            v[i][2*k+j] = new Vertex( slantSide/2 - j*deltaS,
                                      y,
                                      slantSide/2);
         }
         for (int j = 0; j < k; ++j)
         {
            v[i][3*k+j] = new Vertex(-slantSide/2,
                                      y,
                                      slantSide/2 - j*deltaS);
         }
         // create one more vertex to close the latitude
         v[i][4*k] = new Vertex(v[i][0]);
      }

      // Add the vertices to this model.
      for (int i = 0; i < n; ++i)
      {
         for (int j = 0; j < 4*k + 1; ++j)
         {
            addVertex( v[i][j] );
         }
      }
      addVertex(new Vertex(0, h, 0));
      addVertex(new Vertex(0, 0, 0));
      int apexIndex = n * (4*k + 1);
      int bottomCenterIndex = apexIndex + 1;


      // Create the triangle fan at the apex.
      for (int j = 0; j < 4*k; ++j)
      {  //                                     v[n-1][j+1]         v[n-1][j]
         addPrimitive(new Triangle(apexIndex, (n-1)*(4*k+1)+j+1, (n-1)*(4*k+1)+j));
      }

      // Create all the triangle strips around the pyramid wall.
      for (int i = 0; i < n - 1; ++i) // choose a height of latitude
      {
         for (int j = 0; j < 4*k; ++j)
         {  //                          v[i][j]      v[i+1][j]         v[i+1][j+1]
            addPrimitive(new Triangle(i*(4*k+1)+j, (i+1)*(4*k+1)+j, (i+1)*(4*k+1)+j+1));
            //                         v[i+1][j+1]      v[i][j+1]      v[i][j]
            addPrimitive(new Triangle((i+1)*(4*k+1)+j+1, i*(4*k+1)+j+1, i*(4*k+1)+j));
         }
      }

      if (grid)
      {
         // Create the grid in the base of the pyramid.
         // Arrays of indices to be used to create the grid.
         int[][] vGrid = new int[k+1][k+1];
         int vIndex = vertexList.size();
         // Create vertices for the bottom grid.
         double deltaS = s / k;
         for (int i = 0; i < k + 1; ++i)
         {
            for (int j = 0; j < k + 1; ++j)
            {
               addVertex(new Vertex(-s/2 + i * deltaS,
                                     0,
                                    -s/2 + j * deltaS));
               vGrid[i][j] = vIndex;
               vIndex += 1;
            }
         }
         // Create the triangles in the base grid.
         for (int i = 0; i < k; ++i)
         {
            for (int j = 0; j < k; ++j)
            {
               addPrimitive(new Triangle(vGrid[i  ][j  ],
                                         vGrid[i+1][j  ],
                                         vGrid[i+1][j+1]));
               addPrimitive(new Triangle(vGrid[i+1][j+1],
                                         vGrid[i  ][j+1],
                                         vGrid[i  ][j  ]));
            }
         }
      }
      else
      {
         // Create the triangle fan in the base.
         for (int j = 0; j < 4*k; ++j)
         {  //                                           v[0][j]  v[0][j+1]
            addPrimitive(new Triangle(bottomCenterIndex,    j,       j+1 ));
         }
      }
   }
}//Pyramid
