/*

*/

package renderer.models_T;
import  renderer.scene.*;
import  renderer.scene.primitives.*;

/**
   Create a wireframe model of a Sierpinski Sponge centered at the origin.
<p>
   See <a href="https://en.wikipedia.org/wiki/Sierpi%C5%84ski_triangle#Analogues_in_higher_dimensions" target="_top">
                https://en.wikipedia.org/wiki/Sierpi%C5%84ski_triangle#Analogues_in_higher_dimensions</a>
*/
public class SierpinskiSponge extends Model
{
   /**
      Create a Sierpinski Sponge centered at the origin
      using five recursive iterations.
   */
   public SierpinskiSponge()
   {
      this(5);
   }


   /**
      Create a Sierpinski Sponge centered at the origin
      using {@code n} recursive iterations.

      @param n  number of recursive iterations
   */
   public SierpinskiSponge(int n)
   {
      super();

      if (0 == n)
      {
         // Create a tetrahedron geometry.
         // It has 4 vertices and 6 edges.
         Vertex v0 = new Vertex( 1,  1,  1);
         Vertex v1 = new Vertex(-1,  1, -1);
         Vertex v2 = new Vertex( 1, -1, -1);
         Vertex v3 = new Vertex(-1, -1,  1);

         // Add the tetrahedron's vertices to the model.
         addVertex(v0, v1, v2, v3);

         // Create four triangular faces.
         addPrimitive(new Triangle(3, 1, 0),
                      new Triangle(3, 2, 1),
                      new Triangle(3, 0, 2),
                      new Triangle(0, 1, 2));
      }
      else
      {
         nestedModels.add( subSponges(n - 1,  1,  1,  1) );
         nestedModels.add( subSponges(n - 1, -1,  1, -1) );
         nestedModels.add( subSponges(n - 1,  1, -1, -1) );
         nestedModels.add( subSponges(n - 1, -1, -1,  1) );
      }
   }


   /**
      Recursive helper function.
      <p>
      This function builds the four sub models needed
      for one recusive step.

      @param n    number of recursive iterations
      @param pmX  plus or minus 1 for x-direction
      @param pmY  plus or minus 1 for y-direction
      @param pmZ  plus or minus 1 for z-direction
      @return     {@link Model} holding sub tree of sponges
   */
   private Model subSponges(int n, int pmX, int pmY, int pmZ)
   {
      Model model = new Model();
      Matrix scale = Matrix.scale(0.5, 0.5, 0.5);
      Matrix translate = Matrix.translate(pmX*0.5, pmY*0.5, pmZ*0.5);
      model.nestedMatrix = translate.times(scale);
      if (0 == n) // stop the recursion
      {
         // Create a tetrahedron geometry.
         // It has 4 vertices and 6 edges.
         Vertex v0 = new Vertex( 1,  1,  1);
         Vertex v1 = new Vertex(-1,  1, -1);
         Vertex v2 = new Vertex( 1, -1, -1);
         Vertex v3 = new Vertex(-1, -1,  1);

         // Add the tetrahedron's vertices to the model.
         model.addVertex(v0, v1, v2, v3);

         // Create four triangular faces.
         model.addPrimitive(new Triangle(3, 1, 0),
                            new Triangle(3, 2, 1),
                            new Triangle(3, 0, 2),
                            new Triangle(0, 1, 2));
      }
      else
      {
         model.nestedModels.add( subSponges(n - 1,  1,  1,  1) );
         model.nestedModels.add( subSponges(n - 1, -1,  1, -1) );
         model.nestedModels.add( subSponges(n - 1,  1, -1, -1) );
         model.nestedModels.add( subSponges(n - 1, -1, -1,  1) );
      }
      return model;
   }
}//SierpinskiSponge
