/*

*/

package renderer.models_T;
import  renderer.scene.*;
import  renderer.scene.primitives.*;

/**
   Create a solid model of a tetrahedron as a
   triangular pyramid with an equilateral triangle
   base (centered at the origin in the xz-plane)
   whose three vertices are connected to a 4th vertex
   on the positive y-axis.

   @see Tetrahedron
*/
public class TriangularPyramid extends Model
{
   /**
      Create a regular tetrahedron having side length
      {@code sqrt(3)/sqrt(2)}, with one face in the
      xz-plane with its center at the origin, and the
      4th vertex on the positive y-axis at height 1.
   */
   public TriangularPyramid()
   {
      this(Math.sqrt(3)/Math.sqrt(2)); // makes the height = 1
      //or
      //this(Math.sqrt(3));  // make the height = sqrt(2) > 1
   }


   /**
      Create a regular tetrahedron having side length {@code s},
      with one face in the xz-plane with its center at the origin,
      and with the 4th vertex on the positive y-axis at
      height {@code s*sqrt(2)/sqrt(3)}.

      @param s  the length of the regular tetrahedron's sides
   */
   public TriangularPyramid(double s)
   {
      this(s/Math.sqrt(3), s*Math.sqrt(2)/Math.sqrt(3));
   }


   /**
      Create a tetrahedron with one face being an equilateral triangle
      inscribed in a circle of radius {@code r} centered at the origin
      of the xz-plane and with the 4th vertex on the y-axis at height
      {@code h}.
   <p>
      If {@code h = r * sqrt(2)}, then the tetrahedron is a regular tetrahedron.
      with side length {@code s = r * sqrt(3)}.
   <p>
      Another way to state this is, if an equilateral triangle is inscribed
      in a circle of radius {@code r}, then the edge length of the triangle
      is {@code r*sqrt(3)} and the height of the regular tetrahedron made
      from the triangle is {@code r*sqrt(2)}.

      @param r  radius of circle in xz-plane that the equilateral base is inscribed in
      @param h  coordinate on the y-axis of the apex
   */
   public TriangularPyramid(double r, double h)
   {
      super();

      // Create the tetrahedron's geometry.
      double sqrt3 = Math.sqrt(3.0);
      Vertex v0 = new Vertex( r,   0,    0);  // three vertices around the bottom face
      Vertex v1 = new Vertex(-r/2, 0,  r*0.5*sqrt3);
      Vertex v2 = new Vertex(-r/2, 0, -r*0.5*sqrt3);
      Vertex v3 = new Vertex( 0,   h,    0);  // vertex at the top

      addVertex(v0, v1, v2, v3);

      // Create 4 triangles.
      addPrimitive(new Triangle(0, 1, 2),  // bottom face
                   new Triangle(0, 3, 1),  // face 1
                   new Triangle(1, 3, 2),  // face 2
                   new Triangle(0, 2, 3)); // face 3
   }


   /**
      Create a tetrahedron with one face being an equilateral triangle
      inscribed in a circle of radius {@code r} centered at the origin
      of the xz-plane and with the 4th vertex on the y-axis at height
      {@code h}.
   <p>
      If {@code h = r * sqrt(2)}, then the tetrahedron is a regular tetrahedron.
      with side length {@code s = r * sqrt(3)}.
   <p>
      Another way to state this is, if an equilateral triangle is inscribed
      in a circle of radius {@code r}, then the edge length of the triangle
      is {@code r*sqrt(3)} and the height of the regular tetrahedron made
      from the triangle is {@code r*sqrt(2)}.

      @param r  radius of circle in xz-plane that the equilateral base is inscribed in
      @param h  coordinate on the y-axis of the apex
      @param n  number of lines of latitude around the body of the pyramid
      @param k  number of triangles in the triangle fan at the top of each side
   */
   public TriangularPyramid(double r, double h, int n, int k)
   {
      super();

      if (n < 1) n = 1;
      if (k < 1) k = 1;

      // Create the pyramid's geometry.
      // An array of vertices to be used to create faces.
      Vertex[][] v = new Vertex[n][3*k];

      // Create the vertices.
      Vertex apex = new Vertex(0, h, 0);
      Vertex centerVertex = new Vertex(0, 0, 0);

      // three vertices around the bottom face
      double sqrt3 = Math.sqrt(3.0);
      Vertex v0 = new Vertex( r,   0,    0);
      Vertex v1 = new Vertex(-r/2, 0,  r*0.5*sqrt3);
      Vertex v2 = new Vertex(-r/2, 0, -r*0.5*sqrt3);

      for (int i = 0; i < n; i++) // choose a height of latitude
      {
         double y = i * (h / n);

         // use linear interpolation
         double t = i * (1.0 / n);
         Vertex vA = new Vertex( (1-t)*v0.x + t*apex.x,
                                                     y,
                                 (1-t)*v0.z + t*apex.z );
         Vertex vB = new Vertex( (1-t)*v1.x + t*apex.x,
                                                     y,
                                 (1-t)*v1.z + t*apex.z );
         Vertex vC = new Vertex( (1-t)*v2.x + t*apex.x,
                                                     y,
                                 (1-t)*v2.z + t*apex.z );

         // use linear interpolation again
         for (int j = 0; j < k; ++j)
         {
            double s = j * (1.0 / k);
            v[i][j] = new Vertex( (1-s)*vA.x + s*vB.x,
                                                    y,
                                  (1-s)*vA.z + s*vB.z );

            v[i][k+j] = new Vertex( (1-s)*vB.x + s*vC.x,
                                                      y,
                                    (1-s)*vB.z + s*vC.z );

            v[i][2*k+j] = new Vertex( (1-s)*vC.x + s*vA.x,
                                                        y,
                                      (1-s)*vC.z + s*vA.z );
         }
      }

      // Add the vertices to this model.
      for (int i = 0; i < n; ++i)
      {
         for (int j = 0; j < 3*k; ++j)
         {
            addVertex( v[i][j] );
         }
      }
      addVertex(apex);
      addVertex(centerVertex);
      int apexIndex = n * 3*k;
      int centerIndex = apexIndex + 1;

      // Create the triangle fan at the apex.
      for (int j = 0; j < 3*k - 1; ++j)
      {  //                                    v[n-1][j+1]       v[n-1][j]
         addPrimitive(new Triangle(apexIndex, (n-1)*(3*k)+j+1, (n-1)*(3*k)+j));
      }
      //                                    v[n-1][0]     v[n-1][3k-1]
      addPrimitive(new Triangle(apexIndex, (n-1)*(3*k)+0, (n-1)*(3*k)+3*k-1));

      // Create all the square strips around the pyramid wall.
      for (int i = 0; i < n - 1; ++i) // choose a height of latitude
      {
         for (int j = 0; j < 3*k - 1; ++j)
         {  //                        v[i][j]    v[i+1][j]       v[i+1][j+1]
            addPrimitive(new Triangle(i*(3*k)+j, (i+1)*(3*k)+j, (i+1)*(3*k)+j+1));
            //                        v[i][j]    v[i+1][j+1]     v[i][j+1]
            addPrimitive(new Triangle(i*(3*k)+j, (i+1)*(3*k)+j+1, i*(3*k)+j+1));
         }
         //                         v[i][3k-1]      v[i+1][3k-1]       v[i+1][0]
         addPrimitive(new Triangle(i*(3*k)+3*k-1, (i+1)*(3*k)+3*k-1, (i+1)*(3*k)+0));
         //                          v[i][3k-1]      v[i+1][0]     v[i][0]
         addPrimitive(new Triangle(i*(3*k)+3*k-1, (i+1)*(3*k)+0, i*(3*k)+0));
      }

      // Create the triangle fan in the base.
      for (int j = 0; j < 3*k - 1; ++j)
      {  //                                     v[0][j]  v[0][j+1]
         addPrimitive(new Triangle(centerIndex,    j,       j+1));
      }
      //                                    v[0][3k-1]  v[0][0]
      addPrimitive(new Triangle(centerIndex,    3*k-1,      0));
   }
}//TriangularPyramid
