/*

*/

package renderer.models_TP;
import  renderer.scene.*;
import  renderer.scene.primitives.*;

/**
   Create a solid model of a cube with its center
   at the origin, having edge length 2, and with its
   corners at {@code (1, 1, 1)}.
<p>
   This version of the cube model has the top and bottom
   faces of the cube cut up by a triangle fan and the
   front, back, right, and left faces cut up by a grid
   of perpendicular lines.
<p>
   Here is a picture showing how the cube's eight corners
   are labeled.
<pre>{@code
                  v[4]
                   +-----------------+ v[5]
                  /|                /|
                /  |              /  |
              /    |            /    |
            /      |          /      |
      v[7] +-----------------+ v[6]  |
           |       |         |       |               y
           |       |         |       |               |
           |       |         |       |               |
           |  v[0] +---------|-------+ v[1]          |
           |      /          |      /                |
           |    /            |    /                  +----. x
           |  /              |  /                   /
           |/                |/                    /
           +-----------------+                    /
          v[3]              v[2]                 z
}</pre>

   @see Cube
   @see Cube2
   @see Cube3
*/
public class Cube4 extends Model
{
   /**
      Create a cube with its center at the origin, having edge
      length 2, with its corners at {@code (1, 1, 1)}. with
      a triangle fan of eight triangles in top and bottom faces,
      and two perpendicular lines cutting each of the front,
      back, right, and left faces.
   */
   public Cube4( )
   {
      this(2, 1, 2);
   }


   /**
      Create a cube with its center at the origin, having edge
      length 2, with its corners at {@code (1, 1, 1)}, and
      with the top and bottom faces containing a triangle fan
      with the given number of triangles along each of the x,
      and z directions.
      <p>
      There must be at least one triangle along the x and z directions.

      @param xCount  number of triangles along the x-direction
      @param yGrid   number of grid lines perpendicular to the y-axis
      @param zCount  number of triangles along the z-direction
   */
   public Cube4(int xCount, int yGrid, int zCount)
   {
      super();

      if (xCount < 1) xCount = 1;
      if (yGrid  < 0) yGrid  = 0;
      if (zCount < 1) zCount = 1;

      double xStep = 2.0 / xCount;
      double yStep = 2.0 / (1 + yGrid);
      double zStep = 2.0 / zCount;

      // An array of vertices to be used to create faces.
      Vertex[][] v = new Vertex[2 + yGrid][2*xCount + 2*zCount];

      // Create all the vertices.
      for (int i = 0; i < 2 + yGrid; ++i) // choose a height of latitude
      {
         for(int j = 0; j < xCount; ++j)
         {
            v[i][j] = new Vertex(-1 + j*xStep, // move right
                                 -1 + i*yStep,
                                 -1);
         }
         for(int j = 0; j < zCount; ++j)
         {
            v[i][xCount + j] = new Vertex( 1,
                                          -1 + i*yStep,
                                          -1 + j*zStep); // move forward
         }
         for(int j = 0; j < xCount; ++j)
         {
            v[i][xCount+zCount + j] = new Vertex( 1 - j*xStep, // move left
                                                 -1 + i*yStep,
                                                  1);
         }
         for(int j = 0; j < zCount; ++j)
         {
            v[i][2*xCount+zCount + j] = new Vertex(-1,
                                                   -1 + i*yStep,
                                                    1 - j*zStep); // backwards
         }
      }

      // Add all of the vertices to this model.
      for (int i = 0; i < 2 + yGrid; ++i)
      {
         for (int j = 0; j < 2*xCount + 2*zCount; ++j)
         {
            addVertex( v[i][j] );
         }
      }
      addVertex(new Vertex(0, -1,  0)); // center bottom
      addVertex(new Vertex(0,  1,  0)); // center top
      int cBottom = (2 + yGrid) * (2*xCount + 2*zCount);
      int cTop = cBottom + 1;

      for (int j = 0; j < 2*xCount + 2*zCount - 1; ++j)
      {
         Primitive triangles = new Triangles();

         // Create the triangle fans in the bottom and top sides.
         triangles.addIndex(
                  cBottom, j,       // v[0][j]
                           j+1,     // v[0][j+1]
                  cTop, (yGrid+1)*(2*xCount + 2*zCount)+j+1, // v[2+yGrid-1][j+1]
                        (yGrid+1)*(2*xCount + 2*zCount)+j); // v[2+yGrid-1][j]

         // Create the square strips down the cube's vertical sides.
         for (int i = 0; i < 2 + yGrid - 1; ++i) // choose a height of latitude
         {
            triangles.addIndex(
                             i*(2*xCount+2*zCount)+j,   // v[i  ][j  ]
                        (i+1)*(2*xCount+2*zCount)+j,    // v[i+1][j  ]
                        (i+1)*(2*xCount+2*zCount)+j+1); // v[i+1][j+1]

            triangles.addIndex(
                        (i+1)*(2*xCount+2*zCount)+j+1,  // v[i+1][j+1]
                            i*(2*xCount+2*zCount)+j+1,  // v[i  ][j+1]
                            i*(2*xCount+2*zCount)+j);   // v[i  ][j  ]
         }

         addPrimitive(triangles);
      }
      Primitive triangles = new Triangles();
      // Close the triangle fans in the bottom and top sides.
      triangles.addIndex(
            cBottom, 2*xCount + 2*zCount - 1, // v[0][2*xCount + 2*zCount - 1]
                     0,                       // v[0][0]
            cTop, (yGrid+1)*(2*xCount+2*zCount)+0,                   //v[2+yGrid-1][0]
                  (yGrid+1)*(2*xCount+2*zCount)+2*xCount+2*zCount-1);//v[2+yGrid-1][2*xCount+2*zCount-1]

      for (int i = 0; i < 2 + yGrid - 1; ++i) // choose a height of latitude
      {
         triangles.addIndex(
                         i*(2*xCount+2*zCount)+2*xCount+2*zCount-1, // v[i  ][2*xCount+2*zCount-1]
                     (i+1)*(2*xCount+2*zCount)+2*xCount+2*zCount-1, // v[i+1][2*xCount+2*zCount-1]
                     (i+1)*(2*xCount+2*zCount)+0);                  // v[i+1][0]

         triangles.addIndex(
                     (i+1)*(2*xCount+2*zCount)+0,                   // v[i+1][0]
                         i*(2*xCount+2*zCount)+0,                   // v[i  ][0]
                         i*(2*xCount+2*zCount)+2*xCount+2*zCount-1);// v[i  ][2*xCount+2*zCount-1]
      }
      addPrimitive(triangles);
   }
}//Cube4
