/*

*/

package renderer.models_TP;
import  renderer.scene.*;
import  renderer.scene.primitives.*;

/**
   Create a solid model of a frustum of a right square pyramid
   with its base in the xz-plane.
<p>
   See <a href="https://en.wikipedia.org/wiki/Frustum" target="_top">
                https://en.wikipedia.org/wiki/Frustum</a>

   @see Pyramid
*/
public class PyramidFrustum extends Model
{
   /**
      Create a frustum of a right square pyramid with its base in the
      xz-plane, a base side length of 2, top side length of 1, and height 1/2.
   */
   public PyramidFrustum( )
   {
      this(2.0, 1.0, 0.5, 7, 4);
   }


   /**
      Create a frustum of a right square pyramid with its base in the
      xz-plane, a base side length of {@code s1}, top side length of
      {@code s2}, and height {@code h}.
   <p>
      This model works with either {@code s1 > s2} or {@code s1 < s2}.
      In other words, the frustum can have its "apex" either above or
      below the xz-plane.

      @param s1  side length of the base of the frustum
      @param s2  side length of the top of the frustum
      @param h   height of the frustum
   */
   public PyramidFrustum(double s1, double s2, double h)
   {
      super();

      // Create the frustum's geometry.
      Vertex v0 = new Vertex(-s1/2, 0, -s1/2); // 4 vertices around the base
      Vertex v1 = new Vertex(-s1/2, 0,  s1/2);
      Vertex v2 = new Vertex( s1/2, 0,  s1/2);
      Vertex v3 = new Vertex( s1/2, 0, -s1/2);
      Vertex v4 = new Vertex(-s2/2, h, -s2/2); // 4 vertices around the top
      Vertex v5 = new Vertex(-s2/2, h,  s2/2);
      Vertex v6 = new Vertex( s2/2, h,  s2/2);
      Vertex v7 = new Vertex( s2/2, h, -s2/2);
      addVertex(v0, v1, v2, v3, v4, v5, v6, v7);

      // Create 12 triangles.
      addPrimitive(new TriangleStrip(0, 3, 1, 2)); // 2 base triangles
      addPrimitive(new TriangleStrip(4, 0, 5, 1, 6, 2, 7, 3, 4, 0));
      addPrimitive(new TriangleStrip(4, 5, 7, 6)); // 2 top triangles
   }


   /**
      Create a frustum of a right square pyramid with its base in the
      xz-plane, a base side length of {@code s}, top of the frustum at
      height {@code h}, and with the pyramid's apex at on the y-axis at
      height {@code a}.

      @param n  number of lines of latitude
      @param k  number of lines of longitude
      @param s  side length of the base of the frustum
      @param h  height of the frustum
      @param a  height of the apex of the pyramid
   */
   public PyramidFrustum(int n, int k, double s, double h, double a)
   {
      this(s, (1 - h/a)*s, h, n, k);
   }


   /**
      Create a frustum of a right square pyramid with its base in the
      xz-plane, a base side length of {@code s1}, top side length of
      {@code s2}, and height {@code h}.
   <p>
      This model works with either {@code s1 > s2} or {@code s1 < s2}.
      In other words, the frustum can have its "apex" either above or
      below the xz-plane.

      @param s1  side length of the base of the frustum
      @param s2  side length of the top of the frustum
      @param h   height of the frustum
      @param n   number of lines of latitude
      @param k   number of lines of longitude
   */
   public PyramidFrustum(double s1, double s2, double h, int n, int k)
   {
      this(s1, s2, h, n, k, true);
   }


   /**
      Create a frustum of a right square pyramid with its base in the
      xz-plane, a base side length of {@code s1}, top side length of
      {@code s2}, and height {@code h}.
   <p>
      This model works with either {@code s1 > s2} or {@code s1 < s2}.
      In other words, the frustum can have its "apex" either above or
      below the xz-plane.

      @param s1  side length of the base of the frustum
      @param s2  side length of the top of the frustum
      @param h   height of the frustum
      @param n   number of lines of latitude
      @param k   number of lines of longitude
      @param striped  choose between striped and checkerboard triangle strips
   */
   public PyramidFrustum(double s1, double s2, double h, int n, int k, boolean striped)
   {
      super();

      if (n < 2) n = 2;
      if (k < 1) k = 1;

      // Create the pyramid's geometry.
      double deltaH = h / (n - 1);

      // An array of vertices to be used to create faces.
      Vertex[][] v = new Vertex[n][4*k + 1];

      // Create all the vertices.
      for (int i = 0; i < n; ++i) // choose a height of latitude
      {
         double y = i * deltaH;
         double slantSide = s1 - i*((s1 - s2) / (n - 1));
         double deltaS = slantSide / k;

         for (int j = 0; j < k; ++j)
         {
            v[i][j] = new Vertex(-slantSide/2 + j*deltaS,
                                  y,
                                 -slantSide/2);
         }
         for (int j = 0; j < k; ++j)
         {
            v[i][k+j] = new Vertex( slantSide/2,
                                    y,
                                   -slantSide/2 + j*deltaS);
         }
         for (int j = 0; j < k; ++j)
         {
            v[i][2*k+j] = new Vertex( slantSide/2 - j*deltaS,
                                      y,
                                      slantSide/2);
         }
         for (int j = 0; j < k; ++j)
         {
            v[i][3*k+j] = new Vertex(-slantSide/2,
                                     y,
                                     slantSide/2 - j*deltaS);
         }
         // create one more vertex to close the latitude
         v[i][4*k] = new Vertex(v[i][0]);
      }

      // Add all of the vertices to this model.
      for (int i = 0; i < n; ++i)
      {
         for (int j = 0; j < 4*k + 1; ++j)
         {
            addVertex( v[i][j] );
         }
      }
      addVertex(new Vertex(0, h, 0));
      addVertex(new Vertex(0, 0, 0));
      int topCenterIndex = n * (4*k + 1);
      int bottomCenterIndex = topCenterIndex + 1;


      // Create the triangle fan in the base.
      Primitive botFan = new TriangleFan();
      botFan.addIndex(bottomCenterIndex);
      for (int j = 0; j < 4*k + 1; ++j)
      {
         botFan.addIndex( j );  // v[0][j]
      }
      addPrimitive(botFan);

      // Create all the square strips around the pyramid wall.
      if (striped) // switch between striped and checkerboard triangle strips
      {
         for (int i = 0; i < n - 1; ++i) // choose a height of latitude
         {
            Primitive triStrip = new TriangleStrip();
            for (int j = 0; j < 4*k + 1; ++j)
            {
               triStrip.addIndex(    i*(4*k+1)+j); // v[i  ][j]
               triStrip.addIndex((i+1)*(4*k+1)+j); // v[i+1][j]
            }
            addPrimitive(triStrip);
         }
      }
      else // checkerboard triangle strips
      {
         for (int i = 0; i < n - 1; ++i) // choose a height of latitude
         {
            for (int j = 0; j < 4*k; ++j)
            {
               addPrimitive(
                  new TriangleStrip(
                         i*(4*k+1)+j,     // v[i  ][j  ]
                     (i+1)*(4*k+1)+j,     // v[i+1][j  ]
                         i*(4*k+1)+j+1,   // v[i  ][j+1]
                     (i+1)*(4*k+1)+j+1)); // v[i+1][j+1]
            }
         }
      }

      // Create the triangle fan in the top.
      Primitive topFan = new TriangleFan();
      topFan.addIndex(topCenterIndex);
      for (int j = 4*k; j >= 0; --j)
      {
         topFan.addIndex( (n-1)*(4*k+1)+j ); // v[n-1][j]
      }
      addPrimitive(topFan);
   }
}//PyramidFrustum
