/*

*/

package renderer.pipeline;
import  renderer.scene.*;
import  renderer.scene.primitives.Primitive;
import  renderer.scene.primitives.Triangle;

import java.util.List;
import java.util.ArrayList;

/**
   This stage is used by the renderer to cull from a {@link Model}'s
   {@link List} of {@link Primitive}s any {@link Triangle} that
   is "back facing".
<p>
   For a good explanation of orientation, forward and backward facing
   polygons, and back face culling, see Chapter 9 of<br>
      "Computer Graphics Through OpenGL: From Theory to Experiments,"
      3rd Edition, by Sumanta Guha, Chapman and Hall/CRC, 2019.
<p>
   That chapter is available online at<br>
       <a href="http://www.sumantaguha.com/files/materials/ch9.pdf" target="_top">
                http://www.sumantaguha.com/files/materials/ch9.pdf</a>
*/
public class BackFaceCulling2
{
   /**
      Replace the given {@link Model}'s list of primitives with a
      new primitive list from which all back facing {@link Triangle}s
      have been filtered out. Notice that non {@link Triangle}
      primitives are considered to not be back facing, so they are
      kept in the primitive list.

      @param model  {@link Model} whose {@Primitive} list is to be culled
   */
   public static void cull(Model model)
   {
      List<Primitive> frontPrimitives = new ArrayList<>();
      List<Primitive> backPrimitives = new ArrayList<>();
      for (Primitive p : model.primitiveList)
      {
         if (! isBackFace(p, model))
         {
            // Keep the non-triangle primitives and the
            // triangles with the desired orientation.
            frontPrimitives.add(p);
         }
         else if (!model.doBackFaceCulling && model.facesHaveTwoSides)
         {
            // Keep a back facing Triangle but
            // use its back facing colors.
            p.cIndexList = ((Triangle)p).cIndexList2;
            backPrimitives.add(p);
         }
      }
      List<Primitive> primitiveList = new ArrayList<>();
      // put the back facing primitives before the front facing
      // ones so that the front facing ones get drawn last
      for (Primitive p : backPrimitives)
         primitiveList.add(p);
      for (Primitive p : frontPrimitives)
         primitiveList.add(p);
      model.primitiveList = primitiveList;
   }


   /**
      Determine if a {@Primitive} is a {@link Triangle}
      and if it is backward facing. Whether it is back facing or not
      depends on the {@link Model}'s setting for {@code frontFacingIsCCW}.

      @param p      {@link Primitive} with at least three vertices
      @param model  {@link Model} that {@code p} comes from
      @return true if {@code p} is a back facing {@link Triangle}
   */
   public static boolean isBackFace(Primitive p, Model model)
   {
      return ((p instanceof Triangle)
             && (model.frontFacingIsCCW ^ isCCW(p, model)));
   //https://docs.oracle.com/javase/specs/jls/se6/html/expressions.html#15.22.2
   }


   /**
      Determine if a {@link Triangle} is counterclockwise.

      @param p      {@link Primitive} with at least three vertices
      @param model  {@link Model} that {@code p} comes from
      @return true if the {@link Primitive} is counterclockwise
   */
   private static boolean isCCW(Primitive p, Model model)
   {
      // We only need to know the z-component of the cross product
      // of the two edge vectors, v1-v0 and v2-v0. This works
      // because the camera is looking down the negative z-axis.
      double zValueOfCrossProduct;

      double x0 = model.vertexList.get(p.vIndexList.get(0)).x;
      double x1 = model.vertexList.get(p.vIndexList.get(1)).x;
      double x2 = model.vertexList.get(p.vIndexList.get(2)).x;
      double y0 = model.vertexList.get(p.vIndexList.get(0)).y;
      double y1 = model.vertexList.get(p.vIndexList.get(1)).y;
      double y2 = model.vertexList.get(p.vIndexList.get(2)).y;

      double w0 = model.vertexList.get(p.vIndexList.get(0)).w;
      double w1 = model.vertexList.get(p.vIndexList.get(1)).w;
      double w2 = model.vertexList.get(p.vIndexList.get(2)).w;

      // This works in the camera's image plane, but not in
      // homogeneous clip coordinates.
      //zValueOfCrossProduct = (x1-x0)*(y2-y0) - (y1-y0)*(x2-x0);

      // This works in homogeneous clip coordinates, but it has too
      // many division opperations. This is essentially the same as
      // the previous version, but with "perspective division".
      //zValueOfCrossProduct = (x1/w1 - x0/w0) * (y2/w2 - y0/w0)
      //                     - (y1/w1 - y0/w0) * (x2/w2 - x0/w0);

      // This works in homogeneous clip coordinates and it has
      // fewer division opperations.
      //zValueOfCrossProduct = (w0*x1/w1 - x0) * (w0*y2/w2 - y0)
      //                     - (w0*y1/w1 - y0) * (w0*x2/w2 - x0);

      // This works in homogeneous clip coordinates and it doesn't
      // have division opperations.
      //zValueOfCrossProduct = (w0*x1*w2 - x0*w1*w2) * (w0*w1*y2 - y0*w1*w2)
      //                     - (w0*y1*w2 - y0*w1*w2) * (w0*w1*x2 - x0*w1*w2);

      // Here is the previous version with a bit of optimization.
      //zValueOfCrossProduct = (w0*x1 - x0*w1)*w2 * (w0*y2 - y0*w2)*w1
      //                     - (w0*y1 - y0*w1)*w2 * (w0*x2 - x0*w2)*w1;

      // Some more optimization.
      zValueOfCrossProduct = ((w0*x1 - x0*w1) * (w0*y2 - y0*w2)
                            - (w0*y1 - y0*w1) * (w0*x2 - x0*w2)) * w1*w2;

      if (Pipeline.debug)
      {
         System.err.printf("zValueOfCrossProduct = % .26f\n",
                            zValueOfCrossProduct);
      }

      return 0 <= zValueOfCrossProduct;
   }
}
