/*

*/

package renderer.pipeline;
import  renderer.scene.*;
import  renderer.scene.primitives.*;

import java.util.List;
import java.util.ArrayList;
import java.awt.Color;

/**
   Clip a (projected) {@link LineSegment} that sticks out of the
   standard homogeneous clipping region in clip coordinates.
   Interpolate {@link Vertex} color from any clipped off
   {@code Vertex} to the new {@code Vertex}.
<p>
   This clipping algorithm is a simplification of the Liang-Barsky
   Parametric Line Clipping algorithm.
<p>
   This algorithm assumes that all {@code LineSegment} objects have been
   projected into 4-dimensional homogeneous clip coordinates. This
   algorithm also assumes that the standard homogeneous clipping region
   is
   <pre>{@code
      -1 <= x/w <= +1
      -1 <= y/w <= +1
      -1 <= z/w <= +1.
   }</pre>
<p>
   This algorithm uses homogeneous clip coordinates, where the homogeneous
   coordinate {@code w} need NOT be 1. In order to avoid doing a lot of
   unneeded divisions, we will write the inequalities for the standard
   clipping region as
   <pre>{@code
      -w <= x <= w
      -w <= y <= w
      -w <= z <= w.
   }</pre>
   It is important to realize that these homogeneous clipping inequalities
   are correct for both the projection of the  perspective view volume and
   the projection of the  othographic view volume. (These two projected view
   volumes do not equal each other in homogeneous coordinates, but the two
   view volumes will be equal to each other after perspective division.)
<p>
   If a line segment's projected vertex has an {@code x} or {@code y}
   coordinate with absolute value greater than {@code |w|}, then that vertex
   "sticks out" of the standard homgeneous clipping region. This algorithm
   will clip the line segment so that both of the line segment's vertices
   are within the standard clipping region with {@code -w <= x <= w},
   {@code -w <= y <= w} and {@code -w <= z <= w}.
<p>
   NOTE: Rewriting the inequality {@code -1 <= x/w <= +1} as
   {@code -w <= x <= w} only makes sense if we know that {@code w} is
   a positive number. The perspective projection matrix takes points
   {@code (x,y,z,1)} in camera coordinates and transforms them as
   {@code (x,y,z,1) -> (x,y,1,-z)}. But in camera coordinates, {@code z}
   should be negative (why?), so {@code -z} should be positive, so the
   w-coordinates after the perspective projection should be positive.
   (And the orthographic projection matrix transforms camera coordinates
   as {@code (x,y,z,1) -> (x,y,z,1)}, so {@code w} is always positive.)
<p>
   Here is an outline of the clipping algorithm.
<p>
   Recursively process each line segment, using the following steps.
<p>
     1) Test if the line segment no longer needs to be clipped, i.e., both
        of its vertices are within the clipping region. If this is the
        case, then return the line segment.
        <pre>{@code
               x+w=0      x-w=0
                 |          |
                 |          |
             ----+----------+----- y-w = 0
                 |     v1   |
                 |    /     |
                 |   /      |
                 |  /       |
                 | v0       |
             ----+----------+----- y+w = 0
                 |          |
                 |          |
        }</pre>
<p>
     2) Test if the line segment should be "trivially rejected". A line
        segment is "trivially rejected" if it is on the wrong side of any
        of the four planes that bound the clipping region (i.e., the four
        planes {@code x = w}, {@code x = -w}, {@code y = w}, {@code y = -w}).
        If so, then return an empty list (so the line segment will not be
        rasterized into the framebuffer).
<p>
        Notice that a line like the following one is trivially rejected
        because it is on the "wrong" side of the plane {@code x = w}.
        <pre>{@code
                          x-w=0
                            |            v1
                            |            /
                 +----------+           /
                 |          |          /
                 |          |         /
                 |          |        /
                 |          |       /
                 |          |      /
                 +----------+     /
                            |    /
                            |  v0
        }</pre>
        But the following line is NOT trivially rejected because, even
        though it is completely outside of the clipping region, this line
        is not entirely on the wrong side of any one of the four planes
        {@code x = w}, {@code x = -w}, {@code y = w}, or {@code y = -w}.
        The line below will get clipped at least one time (either on the
        plane {@code x = w} or the plane {@code y = -w}) before it is
        (recursively) a candidate for "trivial rejection". Notice that
        the line below could even be clipped twice, first on {@code y = w},
        then on {@code x = w}, before it can be trivially rejected (by
        being on the wrong side of {@code y = -w}).
        <pre>{@code
                          x-w=0
                            |          v1
                            |         /
                 +----------+        /
                 |          |       /
                 |          |      /
                 |          |     /
                 |          |    /
                 |          |   /
                 +----------+  /
                            | /
                            |/
                            /
                           /|
                          / |
                        v0
        }</pre>
<p>
     3) If the line segment has been neither accepted nor rejected, then
        it needs to be clipped. So we test the line segment against each
        of the four clipping planes, {@code x = w}, {@code x = -w},
        {@code y = w}, and {@code y = -w}, to determine if the line segment
        crosses one of those planes. We clip the line segment against the
        first plane which we find that it crosses. Then we recursively clip
        the resulting clipped line segment. Notice that we only clip against
        the first clipping plane which the segment is found to cross. We do
        not continue to test against the other clipping planes. This is
        because it may be the case, after just one clip, that the line
        segment is now a candidate for trivial accept or reject. So rather
        than test the line segment against several more clipping planes
        (which may be useless tests) it is more efficient to recursively
        clip the line segment, which will then start with the trivial accept
        or reject tests.
<p>
        When we clip a line segment against a clipping plane, it is always
        the case that one endpoint of the line segment is on the "right"
        side of the clipping plane and the other endpoint of the line segment
        is on the "wrong" side of the clipping plane. In the following picture,
        assume that {@code v0} is on the "wrong" side of the clipping plane
        ({@code x = w}) and {@code v1} is on the "right" side. So {@code v0}
        needs to be clipped off the line segment and replaced by a new
        endpoint.
        <pre>{@code
                            x-w=0
                         v1   |
                           \  |
                            \ |
                             \|
                              \
                              |\
                              | \
                              |  \
                              |   v0
        }</pre>
        Represent points {@code p(t)} on the line segment between {@code v0}
        and {@code v1} with the following parametric equation.
        <pre>{@code
                  p(t) = (1-t) * v0 + t * v1  with  0 <= t <= 1
        }</pre>
        Notice that this equation parameterizes the line segment starting
        with {@code v0} at {@code t=0} (on the "wrong side") and ending
        with {@code v1} at {@code t=1} (on the "right side"). We need to
        find the value of {@code t} when the line segment crosses the
        clipping plane {@code x = w}. Let {@code v0 = (x0, y0, w0)} and let
        {@code v1 = (x1, y1, w1)}. Then the above parametric equation becomes
        the three component equations
        <pre>{@code
                 x(t) = (1-t) * x0 + t * x1,
                 y(t) = (1-t) * y0 + t * y1,
                 w(t) = (1-t) * w0 + t * w1,  with  0 <= t <= 1.
        }</pre>
        Since the clipping plane in this example is {@code x = w}, we need
        to solve the equation {@code x(t) = w(t)} for {@code t}. So we need
        to solve
        <pre>{@code
                  (1-t) * x0 + t * x1 = (1-t) * w0 + t * w1
        }</pre>
        for {@code t}. Here are a few algebra steps.
        <pre>{@code
                  x0 - t * x0 + t * x1 = w0 - t * w0 + t * w1
                    x0 + t * (x1 - x0) = w0 + t * (w1 - w0)
         t * (x1 - x0) - t * (w1 - w0) = w0 - x0
           t * [(x1 - x0) - (w1 - w0)] = w0 - x0
                                     t = (w0 - x0)/[(x1 - x0) - (w1 - w0)]
                                       = (x0 - w0)/[(x0 - x1) - (w0 - w1)]
                                       = (x0 - w0)/[(x0 - w0) - (x1 - w1)]
        }</pre>
        We get similar equations for {@code t} if we clip against the other
        clipping planes ({@code x = -w}, {@code y = w}, or {@code y = -w}) and
        we assume that {@code v0} is on the "wrong side" and {@code v1} is on
        the "right side". With the above value for {@code t}, the new vertex
        {@code p(t)} that replaces {@code v0} can easily be computed.
        <pre>{@code
                            x-w=0
                        v1    |
                          \   |
                           \  |
                            \ |
                              v0=p( (x0-w0)/[(x0-w0) - (x1-w1)] )
                              |
                              |
                              |
         }</pre>
         Finally, the new line segment between {@code v1} and the new
         {@code v0} is recursively clipped so that it can be checked
         once again to see if it can be trivially accepted, trivially
         rejected, or clipped again.
*/
public class ClipLine
{
   public static boolean debug;

   /**
      If the {@link LineSegment} sticks out of the standard homogeneous
      clipping region, then clip it so that it is contained in the
      clipping region.

      @param model  {@link Model} that the {@link LineSegment} {@code ls} comes from
      @param ls     {@link LineSegment} to be clipped
      @return a boolean that indicates if this line segment is within the view rectangle
   */
   public static List<Primitive> clip(Model model, LineSegment ls)
   {
      debug = Clip.debug; // update the debug status

      List<Primitive> result = new ArrayList<>();

      // Make local copies of several values.
      Vertex v0 = model.vertexList.get( ls.vIndexList.get(0) );
      Vertex v1 = model.vertexList.get( ls.vIndexList.get(1) );

      double x0 = v0.x,  y0 = v0.y,  z0 = v0.z,  w0 = v0.w;
      double x1 = v1.x,  y1 = v1.y,  z1 = v1.z,  w1 = v1.w;

      // 1. Check for trivial accept.
      if (! (Math.abs(x0) > Math.abs(w0)
          || Math.abs(y0) > Math.abs(w0)
          || Math.abs(x1) > Math.abs(w1)
          || Math.abs(y1) > Math.abs(w1)
          || Math.abs(z0) > Math.abs(w0)  //Same clipping eqns as in last
          || Math.abs(z1) > Math.abs(w1)))//renderer, but for different reason
      {
         if (debug) System.out.println("-Trivial accept.");
         result.add(ls);
         return result;
      }
      // 2. Check for trivial delete.
      else if ( (w0 - x0 < 0  &&  w1 - x1 < 0) // right of the plane x=w
             || (w0 + x0 < 0  &&  w1 + x1 < 0) // left of the plane x=-w
             || (w0 - y0 < 0  &&  w1 - y1 < 0) // above the plane y=w
             || (w0 + y0 < 0  &&  w1 + y1 < 0) // below the plane y=-w
             || (w0 - z0 < 0  &&  w1 - z1 < 0) // in front of the plane z=w
             || (w0 + z0 < 0  &&  w1 + z1 < 0))// behind the plane z=-w
      {
         if (debug) System.out.println("-Trivial delete.");
         return result;
      }
      // 3. Need to clip this line segment.
      else if (w0 - z0 < 0 || w1 - z1 < 0)  // ls crosses the plane z = w
      {
         if (w1 - z1 < 0)
         {
            if (debug) System.out.println(" Clip off v1 at z = w.");
            // Create a new Vertex between v0 and v1
            interpolateNewVertex(model, ls, 1, 5);
         }
         else // (w0 - z0 < 0)
         {
            if (debug) System.out.println(" Clip off v0 at z = w.");
            // Create a new Vertex between v1 and v0
            interpolateNewVertex(model, ls, 0, 5);
         }
      }
      else if (w0 + z0 < 0 || w1 + z1 < 0)  // ls crosses the plane z = -w
      {
         if (w1 + z1 < 0)
         {
            if (debug) System.out.println(" Clip off v1 at z = -w.");
            // Create a new Vertex between v0 and v1
            interpolateNewVertex(model, ls, 1, 6);
         }
         else // (w0 + z0 < 0)
         {
            if (debug) System.out.println(" Clip off v0 at z = -w.");
            // Create a new Vertex between v1 and v0
            interpolateNewVertex(model, ls, 0, 6);
         }
      }
      else if (w0 - x0 < 0 || w1 - x1 < 0)  // ls crosses the plane x = w
      {
         if (w1 - x1 < 0)
         {
            if (debug) System.out.println("-Clip off v1 at x = 1.");
            // Create a new Vertex between v0 and v1.
            interpolateNewVertex(model, ls, 1, 1);
         }
         else // (w0 - x0 < 0)
         {
            if (debug) System.out.println("-Clip off v0 at x = 1.");
            // Create a new Vertex between v1 and v0.
            interpolateNewVertex(model, ls, 0, 1);
         }
      }
      else if (w0 + x0 < 0 || w1 + x1 < 0)  // ls crosses the plane x = -w
      {
         if (w1 + x1 < 0)
         {
            if (debug) System.out.println("-Clip off v1 at x = -1.");
            // Create a new Vertex between v0 and v1.
            interpolateNewVertex(model, ls, 1, 2);
         }
         else // (w0 + x0 < 0)
         {
            if (debug) System.out.println("-Clip off v0 at x = -1.");
            // Create a new Vertex between v1 and v0.
            interpolateNewVertex(model, ls, 0, 2);
         }
      }
      else if (w0 - y0 < 0 || w1 - y1 < 0)  // ls crosses the plane y = w
      {
         if (w1 - y1 < 0)
         {
            if (debug) System.out.println("-Clip off v1 at y = 1.");
            // Create a new Vertex between v0 and v1.
            interpolateNewVertex(model, ls, 1, 3);
         }
         else // (w0 - y0 < 0)
         {
            if (debug) System.out.println("-Clip off v0 at y = 1.");
            // Create a new Vertex between v1 and v0.
            interpolateNewVertex(model, ls, 0, 3);
         }
      }
      else if (w0 + y0 < 0 || w1 + y1 < 0)  // ls crosses the plane y = -w
      {
         if (w1 + y1 < 0)
         {
            if (debug) System.out.println("-Clip off v1 at y = -1.");
            // Create a new Vertex between v0 and v1.
            interpolateNewVertex(model, ls, 1, 4);
         }
         else // (w0 + y0 < 0)
         {
            if (debug) System.out.println("-Clip off v0 at y = -1.");
            // Create a new Vertex between v1 and v0.
            interpolateNewVertex(model, ls, 0, 4);
         }
      }
      else // We should never get here.
      {
         System.err.println("Clipping Error!");
         Thread.dumpStack();
       //System.err.println(Arrays.toString(Thread.currentThread().getStackTrace()));
         System.exit(-1);
      }
      return clip(model, ls); // recursively clip this line segment again
   }


   /**
      This method takes in two vertices, one that is on the "right" side
      of a clipping plane and the other that is on the "wrong" side of the
      clipping plane, and an integer which specifies which clipping plane
      to use, where
      <pre>{@code
         eqn_number == 1 means clipping plane x =  w
         eqn_number == 2 means clipping plane x = -w
         eqn_number == 3 means clipping plane y =  w
         eqn_number == 4 means clipping plane y = -w
         eqn_number == 5 means clipping plane z =  w
         eqn_number == 6 means clipping plane z = -w
      }</pre>
      This method returns the vertex that is the intersection point
      between the given line segment and the given clipping line.
      <p>
      This method solves for the value of {@code t} for which the
      parametric equation
      <pre>{@code
                  p(t) = (1-t) * v_outside + t * v_inside
      }</pre>
      intersects the given clipping plane. (Notice that the equation
      is parameterized so that we move from the outside vertex towards
      the inside vertex as {@code t} increases from 0 to 1.) The solved
      for value of {@code t} is then plugged into the parametric formula
      to get the coordinates of the intersection point.

      @param model       {@link Model} that the {@link LineSegment} {@code ls} comes from
      @param ls          the {@link LineSegment} being clipped
      @param outside     the index in {@code ls} of the {@link Vertex} that is outside the clipping region
      @param eqn_number  the identifier of the plane crossed by the line segment {@code ls}
   */
   private static void interpolateNewVertex(Model model,
                                            LineSegment ls,
                                            int outside,
                                            int eqn_number)
   {
      if (debug) System.out.println("-Create new vertex.");

      int inside = 1 - outside;

      // Make local copies of several values.
      // "i" for "inside"
      double vix = model.vertexList.get( ls.vIndexList.get(inside) ).x;
      double viy = model.vertexList.get( ls.vIndexList.get(inside) ).y;
      double viz = model.vertexList.get( ls.vIndexList.get(inside) ).z;
      double viw = model.vertexList.get( ls.vIndexList.get(inside) ).w;
      float ci[] =  model.colorList.get( ls.cIndexList.get(inside) ).getRGBColorComponents(null);
      // and "o" for "outside"
      double vox = model.vertexList.get( ls.vIndexList.get(outside) ).x;
      double voy = model.vertexList.get( ls.vIndexList.get(outside) ).y;
      double voz = model.vertexList.get( ls.vIndexList.get(outside) ).z;
      double vow = model.vertexList.get( ls.vIndexList.get(outside) ).w;
      float co[] =  model.colorList.get( ls.cIndexList.get(outside) ).getRGBColorComponents(null);

      // Interpolate between v_outside and v_inside.
      double t = 0.0;
      if (1 == eqn_number)             // clip to plane x = w
         t = -(vow - vox)/( (viw - vix) - (vow - vox) );
      else if (2 == eqn_number)        // clip to plane x = -w
         t = -(vow + vox)/( (viw + vix) - (vow + vox) );
      else if (3 == eqn_number)       // clip to plane y = w
         t = -(vow - voy)/( (viw - viy) - (vow - voy) );
      else if (4 == eqn_number)       // clip to plane y = -w
         t = -(vow + voy)/( (viw + viy) - (vow + voy) );
      else if (5 == eqn_number)       // clip to plane z = w
         t = -(vow - voz)/( (viw - viz) - (vow - voz) );
      else if (6 == eqn_number)       // clip to plane z = -w
         t = -(vow + voz)/( (viw + viz) - (vow + voz) );

      // Use the value of t to interpolate the coordinates of the new vertex.
      double x = (1-t) * vox + t * vix;
      double y = (1-t) * voy + t * viy;
      double z = (1-t) * voz + t * viz;
      double w = (1-t) * vow + t * viw;

      Vertex v_new = new Vertex(x, y, z, w);

      // Use the value of t to interpolate the color of the new vertex.
      float t_ = (float)t;
      float r = (1-t_) * co[0] + t_ * ci[0];
      float g = (1-t_) * co[1] + t_ * ci[1];
      float b = (1-t_) * co[2] + t_ * ci[2];

      Color c_new = new Color(r, g, b);

      // Modify the Model and LineSegment to contain the new Vertex and Color.
      int vIndex = model.vertexList.size();
      int cIndex = model.colorList.size();
      model.vertexList.add(v_new);
      model.colorList.add(c_new);
      ls.vIndexList.set(outside, vIndex);
      ls.cIndexList.set(outside, cIndex);

      if (debug)
      {
         System.out.printf("  t = % .25f\n", t);
         System.out.printf("  <x_o,y_o,z_o,w_o> = <% .24f % .24f\n", vox,voy);
         System.out.printf("                       % .24f % .24f>\n",voz,vow);
         System.out.printf("  <x_i,y_i,z_i,w_i> = <% .24f % .24f\n", vix,viy);
         System.out.printf("                       % .24f % .24f>\n",viz,viw);
         System.out.printf("  <x,  y,  z,  w>   = <% .24f % .24f\n",   x,  y);
         System.out.printf("                       % .24f % .24f>\n",  z,  w);
         System.out.printf("  <r_o,g_o,b_o> = <% .15f  % .15f  % .15f>\n",
                              co[0], co[1], co[2]);
         System.out.printf("  <r_i,g_i,b_i> = <% .15f  % .15f  % .15f>\n",
                              ci[0], ci[1], ci[2]);
         System.out.printf("  <r,  g,  b>   = <% .15f  % .15f  % .15f>\n",
                               r,  g,  b);
      }
   }
}
