/*

*/

package renderer.pipeline;
import  renderer.scene.*;
import  renderer.scene.primitives.*;

import java.util.List;
import java.util.ArrayList;
import java.awt.Color;

/**
   Clip a (projected) {@link Triangle} that sticks out of
   the standard homogeneous clipping region in clip coordinates.
   Interpolate {@link Vertex} color from any clipped off
   {@code Vertex} to the new {@code Vertex}.
*/
public class ClipTriangle
{
   public static boolean debug;

   /**
      If the {@link Triangle} sticks out of the homogeneous clipping region,
      then clip it so that it is contained in the homogeneous clipping region.
      <p>
      When we clip a triangle, we may end up with as many as ten
      new sub-triangles. The new, clipped, sub-triangles are
      gathered into the return {@link List}.

      @param model  {@link Model} that the {@link Triangle} {@code t} comes from
      @param t      {@link Triangle} to be clipped
      @return a {@link List} that is either empty or contains clipped {@code Triangle}s
   */
   public static List<Primitive> clip(Model model, Triangle t)
   {
      debug = Clip.debug; // update the debug status

      // Create an empty Primitive list to hold
      // the clipped Triangle objects.
      List<Primitive> doneList = new ArrayList<>();

      // Create an empty Triangle list to hold Triangle
      // objects that still need clipping.
      List<Triangle> toDoList = new ArrayList<>();

      toDoList.add(t);

      while ( ! toDoList.isEmpty() )
      {
         Triangle t2 = toDoList.remove(0);

         // Make local copies of several values.
         int index0 = t2.vIndexList.get(0);
         int index1 = t2.vIndexList.get(1);
         int index2 = t2.vIndexList.get(2);

         Vertex v0 = model.vertexList.get( index0 );
         Vertex v1 = model.vertexList.get( index1 );
         Vertex v2 = model.vertexList.get( index2 );

         double x0 = v0.x,  y0 = v0.y, z0 = v0.z, w0 = v0.w;
         double x1 = v1.x,  y1 = v1.y, z1 = v1.z, w1 = v1.w;
         double x2 = v2.x,  y2 = v2.y, z2 = v2.z, w2 = v2.w;

         // 1. Check for trivial accept.
         if (! (Math.abs(x0) > Math.abs(w0)
             || Math.abs(y0) > Math.abs(w0)
             || Math.abs(x1) > Math.abs(w1)
             || Math.abs(y1) > Math.abs(w1)
             || Math.abs(x2) > Math.abs(w2)
             || Math.abs(y2) > Math.abs(w2)
             || Math.abs(z0) > Math.abs(w0)   // Same clipping eqns as
             || Math.abs(z1) > Math.abs(w1)   // in last renderer, but
             || Math.abs(z2) > Math.abs(w2))) // for a different reason.
         {
            // This triangle doesn't need clipping, so put it in the done list.
            if (debug) System.out.println(" Trivial accept.");
            doneList.add(t2);
         }
         // 2. Check for trivial delete.
         else if ( (w0 - x0 < 0  &&  w1 - x1 < 0  &&  w2 - x2 < 0)//right of x=w
                || (w0 + x0 < 0  &&  w1 + x1 < 0  &&  w2 + x2 < 0)//left of x=-w
                || (w0 - y0 < 0  &&  w1 - y1 < 0  &&  w2 - y2 < 0)//above y=w
                || (w0 + y0 < 0  &&  w1 + y1 < 0  &&  w2 + y2 < 0)//below y=-w
                || (w0 - z0 < 0  &&  w1 - z1 < 0  &&  w2 - z2 < 0)//front of z=w
                || (w0 + z0 < 0  &&  w1 + z1 < 0  &&  w2 + z2 < 0))//behind z=-w
         {
            if (debug) System.out.println(" Trivial delete.");
         }
         // 3. Need to clip this triangle.
         else if (w0 - z0 < 0 || w1 - z1 < 0 || w2 - z2 < 0) //crosses z=w
         {
            // Clip to the z = w plane.
            //if (debug) System.out.println(" Clip to z = +w");
            if (w0 - z0 < 0 && w1 - z1 < 0)
            {// Create one new Triangle containing previous v2 and new v0, v1
               if (debug) System.out.println(" Clip off v0 & v1 at z = w.");
               interpolateNewTriangle(model, t2, 2, 0, 1, toDoList, 5);
            }
            else if (w0 - z0 < 0 && w2 - z2 < 0)
            {// Create one new Triangle containing previous v1 and new v0, v2.
               if (debug) System.out.println(" Clip off v0 & v2 at z = w.");
               interpolateNewTriangle(model, t2, 1, 2, 0, toDoList, 5);
            }
            else if (w1 - z1 < 0 && w2 - z2 < 0)
            {// Create one new Triangle containing previous v0 and new v1, v2.
               if (debug) System.out.println(" Clip off v1 & v2 at z = w.");
               interpolateNewTriangle(model, t2, 0, 1, 2, toDoList, 5);
            }
            else if (w0 - z0 < 0)
            {// Create two new Triangles, neither containing previous v0.
               if (debug) System.out.println(" Clip off v0 at z = w.");
               interpolateNewTriangles(model, t2, 0, 1, 2, toDoList, 5);
            }
            else if (w1 - z1 < 0)
            {// Create two new Triangles, neither containing previous v1.
               if (debug) System.out.println(" Clip off v1 at z = w.");
               interpolateNewTriangles(model, t2, 1, 2, 0, toDoList, 5);
            }
            else if (w2 - z2 < 0)
            {// Create two new Triangles, neither containing previous v2.
               if (debug) System.out.println(" Clip off v2 at z = w.");
               interpolateNewTriangles(model, t2, 2, 0, 1, toDoList, 5);
            }
            else  // We should never get here.
            {
               System.err.println("Clipping to z = +w dropped a triangle!");
               Thread.dumpStack();
               System.exit(-1);
            }
         }
         else if (w0 + z0 < 0 || w1 + z1 < 0 || w2 + z2 < 0) //crosses z=-w
         {
            // Clip to the z = -w plane.
            //if (debug) System.out.println(" Clip to z = -w");
            if (w0 + z0 < 0 && w1 + z1 < 0)
            {// Create one new Triangle containing previous v2 and new v0, v1
               if (debug) System.out.println(" Clip off v0 & v1 at z = -w.");
               interpolateNewTriangle(model, t2, 2, 0, 1, toDoList, 6);
            }
            else if (w0 + z0 < 0 && w2 + z2 < 0)
            {// Create one new Triangle containing previous v1 and new v0, v2.
               if (debug) System.out.println(" Clip off v0 & v2 at z = -w.");
               interpolateNewTriangle(model, t2, 1, 2, 0, toDoList, 6);
            }
            else if (w1 + z1 < 0 && w2 + z2 < 0)
            {// Create one new Triangle containing previous v0 and new v1, v2.
               if (debug) System.out.println(" Clip off v1 & v2 at z = -w.");
               interpolateNewTriangle(model, t2, 0, 1, 2, toDoList, 6);
            }
            else if (w0 + z0 < 0)
            {// Create two new Triangles, neither containing previous v0.
               if (debug) System.out.println(" Clip off v0 at z = -w.");
               interpolateNewTriangles(model, t2, 0, 1, 2, toDoList, 6);
            }
            else if (w1 + z1 < 0)
            {// Create two new Triangles, neither containing previous v1.
               if (debug) System.out.println(" Clip off v1 at z = -w.");
               interpolateNewTriangles(model, t2, 1, 2, 0, toDoList, 6);
            }
            else if (w2 + z2 < 0)
            {// Create two new Triangles, neither containing previous v2.
               if (debug) System.out.println(" Clip off v2 at z = -w.");
               interpolateNewTriangles(model, t2, 2, 0, 1, toDoList, 6);
            }
            else  // We should never get here.
            {
               System.err.println("Clipping to z = -w dropped a triangle!");
               Thread.dumpStack();
               System.exit(-1);
            }
         }
         else if (w0 - x0 < 0 || w1 - x1 < 0 || w2 - x2 < 0) //crosses x=w
         {
            // Clip to the x = w plane.
            //if (debug) System.out.println(" Clip to x = +1");
            if (w0 - x0 < 0 && w1 - x1 < 0)
            {// Create one new Triangle containing previous v2 and new v0, v1.
               if (debug) System.out.println(" Clip off v0 & v1 at x = w.");
               interpolateNewTriangle(model, t2, 2, 0, 1, toDoList, 1);
            }
            else if (w0 - x0 < 0 && w2 - x2 < 0)
            {// Create one new Triangle containing previous v1 and new v0, v2.
               if (debug) System.out.println(" Clip off v0 & v2 at x = w.");
               interpolateNewTriangle(model, t2, 1, 2, 0, toDoList, 1);
            }
            else if (w1 - x1 < 0 && w2 - x2 < 0)
            {// Create one new Triangle containing previous v0 and new v1, v2.
               if (debug) System.out.println(" Clip off v1 & v2 at x = w.");
               interpolateNewTriangle(model, t2, 0, 1, 2, toDoList, 1);
            }
            else if (w0 - x0 < 0)
            {// Create two new Triangles, neither containing previous v0.
               if (debug) System.out.println(" Clip off v0 at x = w.");
               interpolateNewTriangles(model, t2, 0, 1, 2, toDoList, 1);
            }
            else if (w1 - x1 < 0)
            {// Create two new Triangles, neither containing previous v1.
               if (debug) System.out.println(" Clip off v1 at x = w.");
               interpolateNewTriangles(model, t2, 1, 2, 0, toDoList, 1);
            }
            else if (w2 - x2 < 0)
            {// Create two new Triangles, neither containing previous v2.
               if (debug) System.out.println(" Clip off v2 at x = w.");
               interpolateNewTriangles(model, t2, 2, 0, 1, toDoList, 1);
            }
            else  // We should never get here.
            {
               System.err.println("Clipping to x = +w dropped a triangle!");
               Thread.dumpStack();
               System.exit(-1);
            }
         }
         else if (w0 + x0 < 0 || w1 + x1 < 0 || w2 + x2 < 0) //crosses x=-w
         {
            // Clip to the x = -w plane.
            //if (debug) System.out.println(" Clip to x = -1");
            if (w0 + x0 < 0 && w1 + x1 < 0)
            {// Create one new Triangle containing previous v2 and new v0, v1.
               if (debug) System.out.println(" Clip off v0 & v1 at x = -w.");
               interpolateNewTriangle(model, t2, 2, 0, 1, toDoList, 2);
            }
            else if (w0 + x0 < 0 && w2 + x2 < 0)
            {// Create one new Triangle containing previous v1 and new v0, v2.
               if (debug) System.out.println(" Clip off v0 & v2 at x = -w.");
               interpolateNewTriangle(model, t2, 1, 2, 0, toDoList, 2);
            }
            else if (w1 + x1 < 0 && w2 + x2 < 0)
            {// Create one new Triangle containing previous v0 and new v1, v2.
               if (debug) System.out.println(" Clip off v1 & v2 at x = -w.");
               interpolateNewTriangle(model, t2, 0, 1, 2, toDoList, 2);
            }
            else if (w0 + x0 < 0)
            {// Create two new Triangles, neither containing previous v0.
               if (debug) System.out.println(" Clip off v0 at x = -w.");
               interpolateNewTriangles(model, t2, 0, 1, 2, toDoList, 2);
            }
            else if (w1 + x1 < 0)
            {// Create two new Triangles, neither containing previous v1.
               if (debug) System.out.println(" Clip off v1 at x = -w.");
               interpolateNewTriangles(model, t2, 1, 2, 0, toDoList, 2);
            }
            else if (w2 + x2 < 0)
            {// Create two new Triangles, neither containing previous v2.
               if (debug) System.out.println(" Clip off v2 at x = -w.");
               interpolateNewTriangles(model, t2, 2, 0, 1, toDoList, 2);
            }
            else  // We should never get here.
            {
               System.err.println("Clipping to x = -w dropped a triangle!");
               Thread.dumpStack();
               System.exit(-1);
            }
         }
         else if (w0 - y0 < 0 || w1 - y1 < 0 || w2 - y2 < 0) //crosses y=w
         {
            // Clip to the y = +w plane.
            //if (debug) System.out.println(" Clip to y = +1");
            if (w0 - y0 < 0 && w1 - y1 < 0)
            {// create one new Triangle containing previous v2 and new v0, v1.
               if (debug) System.out.println(" Clip off v0 & v1 at y = w.");
               interpolateNewTriangle(model, t2, 2, 0, 1, toDoList, 3);
            }
            else if (w0 - y0 < 0 && w2 - y2 < 0)
            {// Create one new Triangle containing previous v1 and new v0, v2.
               if (debug) System.out.println(" Clip off v0 & v2 at y = w.");
               interpolateNewTriangle(model, t2, 1, 2, 0, toDoList, 3);
            }
            else if (w1 - y1 < 0 && w2 - y2 < 0)
            {// Create one new Triangle containing previous v0 and new v1, v2.
               if (debug) System.out.println(" Clip off v1 & v2 at y = w.");
               interpolateNewTriangle(model, t2, 0, 1, 2, toDoList, 3);
            }
            else if (w0 - y0 < 0)
            {// Create two new Triangles, neither containing previous v0.
               if (debug) System.out.println(" Clip off v0 at y = w.");
               interpolateNewTriangles(model, t2, 0, 1, 2, toDoList, 3);
            }
            else if (w1 - y1 < 0)
            {// Create two new Triangles, neither containing previous v1.
               if (debug) System.out.println(" Clip off v1 at y = w.");
               interpolateNewTriangles(model, t2, 1, 2, 0, toDoList, 3);
            }
            else if (w2 - y2 < 0)
            {// Create two new Triangles, neither containing previous v2.
               if (debug) System.out.println(" Clip off v2 at y = w.");
               interpolateNewTriangles(model, t2, 2, 0, 1, toDoList, 3);
            }
            else  // We should never get here.
            {
               System.err.println("Clipping to y = +w dropped a triangle!");
               Thread.dumpStack();
               System.exit(-1);
            }
         }
         else if (w0 + y0 < 0 || w1 + y1 < 0 || w2 + y2 < 0) //crosses y=-w
         {
            // Clip to the y = -w plane.
            //if (debug) System.out.println(" Clip to y = -w");
            if (w0 + y0 < 0 && w1 + y1 < 0)
            {// Create one new Triangle containing previous v2 and new v0, v1.
               if (debug) System.out.println(" Clip off v0 & v1 at y = -w.");
               interpolateNewTriangle(model, t2, 2, 0, 1, toDoList, 4);
            }
            else if (w0 + y0 < 0 && w2 + y2 < 0)
            {// Create one new Triangle containing previous v1 and new v0, v2.
               if (debug) System.out.println(" Clip off v0 & v2 at y = -w.");
               interpolateNewTriangle(model, t2, 1, 2, 0, toDoList, 4);
            }
            else if (w1 + y1 < 0 && w2 + y2 < 0)
            {// Create one new Triangle containing previous v0 and new v1, v2.
               if (debug) System.out.println(" Clip off v1 & v2 at y = -w.");
               interpolateNewTriangle(model, t2, 0, 1, 2, toDoList, 4);
            }
            else if (w0 + y0 < 0)
            {// Create two new Triangles, neither containing previous v0.
               if (debug) System.out.println(" Clip off v0 at y = -w.");
               interpolateNewTriangles(model, t2, 0, 1, 2, toDoList, 4);
            }
            else if (w1 + y1 < 0)
            {// Create two new Triangles, neither containing previous v1.
               if (debug) System.out.println(" Clip off v1 at y = -w.");
               interpolateNewTriangles(model, t2, 1, 2, 0, toDoList, 4);
            }
            else if (w2 + y2 < 0)
            {// Create two new Triangles, neither containing previous v2.
               if (debug) System.out.println(" Clip off v2 at y = -w.");
               interpolateNewTriangles(model, t2, 2, 0, 1, toDoList, 4);
            }
            else  // We should never get here.
            {
               System.err.println("Clipping to y = -w dropped a triangle!");
               Thread.dumpStack();
               System.exit(-1);
            }
         }
         else // We should never get here.
         {
            System.err.println("Triangle Clipping Error!");
            Thread.dumpStack();
          //System.err.println(Arrays.toString(Thread.currentThread().getStackTrace()));
            System.exit(-1);
         }
      }
      return doneList;
   }


   /**
      This method takes three indices for three vertices, one that is on
      the "right" side of a clipping plane and the other two that are on the
      "wrong" side of the clipping plane, a "todo" {@link List} into which
      a new clipped {@link Triangle} should be put (since it may need to
      be clipped again), and an integer which specifies which clipping line
      to use when calling {@link #interpolateNewVertex}.
      <p>
      When this method returns, it will have placed in {@code triangleList}
      a clipped {@link Triangle} with the index index0_inside and the indices
      for two new interpolated vertices, both of which have been put into
      the {@link Model}'s vertex list.

      @param model           {@link Model} object that holds the vertex list
      @param tri             the {@link Triangle} being clipped
      @param index0_inside   the index in {@code tri} of the {@link Vertex} that is inside the view rectangle
      @param index1_outside  the index in {@code tri} of a {@link Vertex} that needs to be clipped off
      @param index2_outside  the index in {@code tri} of a {@link Vertex} that needs to be clipped off
      @param triangleList    {@link List} to hold new {@link Triangle} object that replaces the clipped {@link Triangle}
      @param eqn_number      the identifier of the line crossed by the {@link Triangle}
   */
   private static void interpolateNewTriangle(Model model,
                                              Triangle tri,
                                              int index0_inside,
                                              int index1_outside,
                                              int index2_outside,
                                              List<Triangle> triangleList,
                                              int eqn_number)
   {
      if (debug) System.out.printf(" Create one new triangle.\n");

      // Interpolate a new vertex between index0_inside and index1_outside.
      // The new veretx and color objects are at the end of the model's
      // vertex and color lists.
      interpolateNewVertex(model,
                           tri,
                           index0_inside,
                           index1_outside,
                           eqn_number);

      // Interpolate a new vertex between index0_inside and index2_outside.
      // The new veretx and color objects are at the end of the model's
      // vertex and color lists.
      interpolateNewVertex(model,
                           tri,
                           index0_inside,
                           index2_outside,
                           eqn_number);

      // Create a new Triangle primitive using the new vertex.
      // (We cannot reuse the original Triangle object since we don't
      //  know which vertex is in which location of the triangle.)
      int vIndex1_new = model.vertexList.size() - 2;
      int cIndex1_new = model.colorList.size() - 2;
      int vIndex2_new = model.vertexList.size() - 1;
      int cIndex2_new = model.colorList.size() - 1;

      Triangle tri_new = new Triangle(tri.vIndexList.get(index0_inside),
                                      vIndex1_new,
                                      vIndex2_new,
                                      tri.cIndexList.get(index0_inside),
                                      cIndex1_new,
                                      cIndex2_new);

      // Add the Triangle to the end of the todo list
      // since the Triangle may need to be clipped again.
      triangleList.add(tri_new);

      //System.out.println( tri_new );
   }


   /**
      This method takes three indices for three vertices, two that are on
      the "right" side of a clipping line and one that is on the "wrong"
      side of the clipping line, a "todo" {@link List} into which two new
      clipped {@link Triangle} objects should be put (since they may need
      to be clipped again), and an integer which specifies which clipping
      line to use when calling {@link #interpolateNewVertex}.
      <p>
      When this method returns, it will have placed in {@code triangleList}
      two new {@link Triangle} objects, one from the two inside vertices to
      a new interpolated vertex, and the other triangle from the two new
      interpolated vertices to one of the original inside vertices. Two new
      {@link Vertex} objects will have also been added to the {@link Model}'s
      vertex list.

      @param model           {@link Model} object that holds the vertex list
      @param tri             the {@link Triangle} being clipped
      @param index0_outside  the index in {@code tri} of the {@link Vertex} that needs to be clipped off
      @param index1_inside   the index in {@code tri} of a {@link Vertex} that is inside the view rectangle
      @param index2_inside   the index in {@code tri} of a {@link Vertex} that is inside the view rectangle
      @param triangleList    {@link List} to hold two new {@link Triangle} objects that replace the clipped {@link Triangle}
      @param eqn_number      the identifier of the line crossed by the triangle
   */
   private static void interpolateNewTriangles(Model model,
                                               Triangle tri,
                                               int index0_outside,
                                               int index1_inside,
                                               int index2_inside,
                                               List<Triangle> triangleList,
                                               int eqn_number)
   {
      if (debug) System.out.printf(" Create two new triangles.\n");

      // Interpolate a new vertex between v1_inside and v0_outside.
      // The new veretx and color objects are at the end of the model's
      // vertex and color lists.
      interpolateNewVertex(model,
                           tri,
                           index1_inside,
                           index0_outside,
                           eqn_number);
      int vIndex0_new1 = model.vertexList.size() - 1;
      int cIndex0_new1 = model.colorList.size() - 1;

      // Interpolate a new vertex between v2_inside and v0_outside.
      // The new veretx and color objects are at the end of the model's
      // vertex and color lists.
      interpolateNewVertex(model,
                           tri,
                           index2_inside,
                           index0_outside,
                           eqn_number);
      int vIndex0_new2 = model.vertexList.size() - 1;
      int cIndex0_new2 = model.colorList.size() - 1;

      // Create a new Triangle primitive using the first new vertex.
      // (We cannot reuse the original Triangle object since we don't
      //  know which vertex is in which location of the triangle.)
      Triangle tri_1 = new Triangle(vIndex0_new1,
                                    tri.vIndexList.get(index1_inside),
                                    tri.vIndexList.get(index2_inside),
                                    cIndex0_new1,
                                    tri.cIndexList.get(index1_inside),
                                    tri.cIndexList.get(index2_inside));

      // Create a new Triangle primitive using the two new vertices.
      Triangle tri_2 = new Triangle(vIndex0_new2,
                                    vIndex0_new1,
                                    tri.vIndexList.get(index2_inside),
                                    cIndex0_new2,
                                    cIndex0_new1,
                                    tri.cIndexList.get(index2_inside));

      // Add the two Triangles to the end of the todo list
      // since the Triangles may need to be clipped again.
      triangleList.add(tri_1);
      triangleList.add(tri_2);

      //System.out.println( tri_1 );
      //System.out.println( tri_2 );
   }

   /**
      This method takes in two vertices, one that is on the "right" side
      of a clipping plane and the other that is on the "wrong" side of the
      clipping plane, and an integer which specifies which clipping plane
      to use, where
      <pre>{@code
         eqn_number == 1 means clipping plane x =  w
         eqn_number == 2 means clipping plane x = -w
         eqn_number == 3 means clipping plane y =  w
         eqn_number == 4 means clipping plane y = -w
         eqn_number == 5 means clipping plane z =  w
         eqn_number == 6 means clipping plane z = -w
      }</pre>
      This method adds to the {@link Model}'s vertex list the vertex
      that is the intersection point between the given line segment
      and the given clipping line.
      <p>
      This method solves for the value of {@code t} for which the
      parametric equation
      <pre>{@code
                  p(t) = (1-t) * v_outside + t * v_inside
      }</pre>
      intersects the given clipping plane. (Notice that the equation
      is parameterized so that we move from the outside vertex towards
      the inside vertex as {@code t} increases from 0 to 1.) The solved
      for value of {@code t} is then plugged into the parametric formula
      to get the coordinates of the intersection point. The solved for
      value of {@code t} is also used to interpolate a new {@link Color}
      object for the new vertex, and the new {@link Color} object is
      added to the end of the {@link Model}'s color list.

      @param model     {@link Model} object that holds the vertex list
      @param tri       the {@link Triangle} being clipped
      @param inside    the index in {@code tri} of the {@link Vertex} that is inside the view rectangle
      @param outside   the index in {@code tri} of the {@link Vertex} that is outside the view rectangle
      @param eqn_number  the identifier of the view rectangle edge crossed by the line segment
   */
   private static void interpolateNewVertex(Model model,
                                            Triangle tri,
                                            int inside,
                                            int outside,
                                            int eqn_number)
   {
      //if (debug) System.out.println(" Create new vertex.");

      // Make local copies of several values.
      Vertex v_inside  = model.vertexList.get( tri.vIndexList.get(inside) );
      Vertex v_outside = model.vertexList.get( tri.vIndexList.get(outside) );
      Color c_inside  = model.colorList.get( tri.cIndexList.get(inside) );
      Color c_outside = model.colorList.get( tri.cIndexList.get(outside) );

      double vix = v_inside.x; // "i" for "inside"
      double viy = v_inside.y;
      double viz = v_inside.z;
      double viw = v_inside.w;
      float ci[] = c_inside.getRGBColorComponents(null);
      double vox = v_outside.x; // "o" for "outside"
      double voy = v_outside.y;
      double voz = v_outside.z;
      double vow = v_outside.w;
      float co[] = c_outside.getRGBColorComponents(null);
      // Interpolate between v_outside and v_inside.
      double t = 0.0;
      if (1 == eqn_number)             // clip to plane x = w
         t = -(vow - vox)/( (viw - vix) - (vow - vox) );
      else if (2 == eqn_number)        // clip to plane x = -w
         t = -(vow + vox)/( (viw + vix) - (vow + vox) );
      else if (3 == eqn_number)       // clip to plane y = w
         t = -(vow - voy)/( (viw - viy) - (vow - voy) );
      else if (4 == eqn_number)       // clip to plane y = -w
         t = -(vow + voy)/( (viw + viy) - (vow + voy) );
      else if (5 == eqn_number)       // clip to plane z = w
         t = -(vow - voz)/( (viw - viz) - (vow - voz) );
      else if (6 == eqn_number)       // clip to plane z = -w
         t = -(vow + voz)/( (viw + viz) - (vow + voz) );

      //if (t < 0.9999999)
      //t = t + 0.0000001;  /*** keep the new vertex off the edge!! ***/

      // Use the value of t to interpolate the coordinates of the new vertex.
    //double x = (1-t) * vox + t * vix;
    //double y = (1-t) * voy + t * viy;
    //double z = (1-t) * voz + t * viz;
    //double w = (1-t) * vow + t * viw;
      double x = vox + t * (vix - vox);
      double y = voy + t * (viy - voy);
      double z = voz + t * (viz - voz);
      double w = vow + t * (viw - vow);

      Vertex v_new = new Vertex(x, y, z, w);

      // Use the value of t to interpolate the color of the new vertex.
    //float r = (1-t) * co[0] + t * ci[0];
    //float g = (1-t) * co[1] + t * ci[1];
    //float b = (1-t) * co[2] + t * ci[2];
      float t_ = (float)t;
      float r = co[0] + t_ * (ci[0] - co[0]);
      float g = co[1] + t_ * (ci[1] - co[1]);
      float b = co[2] + t_ * (ci[2] - co[2]);

      Color c_new = new Color(r, g, b);

      // Modify the Model to contain the new Vertex and Color.
      model.vertexList.add(v_new);
      model.colorList.add(c_new);

      if (debug)
      {
         System.out.printf("  t = % .25f\n", t);
         System.out.printf("  <x_o,y_o,z_o,w_o> = <% .24f % .24f\n", vox,voy);
         System.out.printf("                       % .24f % .24f>\n",voz,vow);
         System.out.printf("  <x_i,y_i,z_i,w_i> = <% .24f % .24f\n", vix,viy);
         System.out.printf("                       % .24f % .24f>\n",viz,viw);
         System.out.printf("  <x,  y,  z,  w>   = <% .24f % .24f\n",   x,  y);
         System.out.printf("                       % .24f % .24f>\n",  z,  w);
         System.out.printf("  <r_o,g_o,b_o> = <% .15f  % .15f  % .15f>\n",
                              co[0], co[1], co[2]);
         System.out.printf("  <r_i,g_i,b_i> = <% .15f  % .15f  % .15f>\n",
                              ci[0], ci[1], ci[2]);
         System.out.printf("  <r,  g,  b>   = <% .15f  % .15f  % .15f>\n",
                               r,  g,  b);
      }
   }
}
