/*

*/

package renderer.pipeline;
import  renderer.scene.*;
import  renderer.scene.primitives.*;
import  renderer.framebuffer.*;

import java.util.List;
import java.util.ArrayList;
import java.awt.Color;

/**
   This renderer takes as its input a {@link Scene} data structure
   and a {@link FrameBuffer} data structure. This renderer mutates
   the {@link FrameBuffer}'s current viewport so that it is filled
   in with the rendered image of the scene represented by the
   {@link Scene} object.
<p>
    This implements our seventeenth rendering pipeline. This renderer
    adds a near (front) clipping plane to the clipping stage. This
    renderer has the same ten pipeline stages as the previous one.
*/
public class Pipeline
{
   public static boolean doAntialiasing = false;
   public static boolean debug = false;

   /**
      Mutate the {@link FrameBuffer}'s current viewport so that
      it holds the rendered image of the {@link Scene} object.

      @param scene  {@link Scene} object to render
      @param fb     {@link FrameBuffer} to hold rendered image of the {@link Scene}
   */
   public static void render(Scene scene, FrameBuffer fb)
   {
      if ( debug )
         render_debug(scene, fb);
      else
         render_(scene, fb);
   }


   /**
      Renderer the {@link Scene} without any debugging information.

      @param scene  {@link Scene} object to render
      @param fb     {@link FrameBuffer} to hold rendered image of the {@link Scene}
   */
   private static void render_(Scene scene, FrameBuffer fb)
   {
      // For every Position in the Scene, render the Position's Model
      // and every nested Position.
      for (Position position : scene.positionList)
      {
         if ( position.visible )
         {
            // Begin a pre-order, depth-first-traversal from this Position.
            render_position(scene, position, Matrix.identity(), fb);
         }
      }
   }


   /**
      Recursively renderer a {@link Position} without any debugging information.
      <p>
      This method does a pre-order, depth-first-traversal of the tree of
      {@link Position}'s rooted at the parameter {@code position}.
      <p>
      The pre-order "visit node" operation in this traversal first updates the
      "current transformation matrix", ({@code ctm}), using the {@link Matrix}
      in {@code position} and then recursively renders the {@link Model} in
      {@code position} using the updated {@code ctm} for the {@link Model2View} stage.

      @param scene     the {@link Scene} that we are rendering
      @param position  the current {@link Position} object to recursively render
      @param ctm       current model-to-view transformation {@link Matrix}
      @param fb        {@link FrameBuffer} to hold rendered image of the {@link Position}
   */
   private static void render_position(Scene scene, Position position, Matrix ctm, FrameBuffer fb)
   {
      // Update the current model-to-view transformation matrix.
      ctm = ctm.times( position.modelMatrix );

      // Render the Position's Model (if it exits and is visible).
      if ( position.model != null && position.model.visible )
      {
         // Do a pre-order, depth-first-traversal from this Model.
         render_model(scene, position.model, ctm, fb);
      }

      // Recursively render every nested Position of this Position.
      for (Position p : position.nestedPositions)
      {
         if ( p.visible )
         {
            // Do a pre-order, depth-first-traversal from this nested Position.
            render_position(scene, p, ctm, fb);
         }
      }
   }


   /**
      Recursively renderer a {@link Model} without any debugging information.
      <p>
      This method does a pre-order, depth-first-traversal of the tree of
      {@link Model}'s rooted at the parameter {@code model}.
      <p>
      The pre-order "visit node" operation in this traversal first updates the
      "current transformation matrix", ({@code ctm}), using the {@link Matrix}
      in {@code model} and then recursively renders {@code model} using the
      updated {@code ctm}.

      @param scene   the {@link Scene} that we are rendering
      @param model   the current {@link Model} object to recursively render
      @param ctm     current model-to-view transformation {@link Matrix}
      @param fb      {@link FrameBuffer} to hold rendered image of the {@link Position}
   */
   private static void render_model(Scene scene, Model model, Matrix ctm, FrameBuffer fb)
   {
      if (model.vertexList.isEmpty() && ! model.primitiveList.isEmpty())
      {
         System.err.println("***WARNING: This model does not have any vertices.");
         System.err.println(model);
      }
      if (! model.vertexList.isEmpty() && model.primitiveList.isEmpty())
      {
         System.err.println("***WARNING: This model does not have any primitives.");
         System.err.println(model);
      }
      if (! model.vertexList.isEmpty() && model.colorList.isEmpty())
      {
         System.err.println("***WARNING: This model does not have any colors.");
         System.err.println(model);
      }

      // Update the current model-to-view transformation matrix.
      ctm = ctm.times( model.nestedMatrix );

      // 0. Make a deep copy of the Model.
      Model model2 = new Model(model);

      // 1. Apply the current model-to-world coordinate transformation.
      Model2World.model2world(model2.vertexList, ctm);

      // 2. Apply the Camera's world-to-view coordinate transformation.
      World2View.world2view(model2.vertexList, scene.camera);

      // 3. Apply the Camera's normalizing view-to-camera coordinate transformation.
      View2Camera.view2camera(model2.vertexList, scene.camera);

      // 4. Apply the camera's projection transformation to homogeneous clip coordinates.
      Projection.project(model2.vertexList, scene.camera);

      // 5. Do the back face culling (of faces) operation.
      if (model2.doBackFaceCulling || model2.facesHaveTwoSides)
      {
         BackFaceCulling1.cull(model2);
      }

      // 6. Do the primitive assembly operation.
      PrimitiveAssembly.assemble(model2);

      // 7. Do the back face culling (of triangles) operation.
      if (model2.doBackFaceCulling || model2.facesHaveTwoSides)
      {
         BackFaceCulling2.cull(model2);
      }

      // 8. Clip each primitive (in homogeneous clip space).
      List<Primitive> primitiveList2 = new ArrayList<>();
      for (Primitive p : model2.primitiveList)
      {
         primitiveList2.addAll( Clip.clip(model2, p) );
      }
      model2.primitiveList = primitiveList2;

      // 9. Apply the perspective division operation to NDC.
      PerspectiveDivision.perspectiveDivide(model2.vertexList);

      // 10. Rasterize each visible primitive into pixels.
      for (Primitive p : model2.primitiveList)
      {
         Rasterize.rasterize(model2, p, fb);
      }

      // Recursively render every nested Model of this Model.
      if (! model2.nestedModels.isEmpty())
      {
         for (Model m : model2.nestedModels)
         {
            if ( m.visible )
            {
               // Do a pre-order, depth-first-traversal from this nested Model.
               render_model(scene, m, ctm, fb);
            }
         }
      }
   }


   /**
      Render the {@link Scene} with quite a bit of debugging information.

      @param scene  {@link Scene} object to render
      @param fb     {@link FrameBuffer} to hold rendered image of the {@link Scene}
   */
   private static void render_debug(Scene scene, FrameBuffer fb)
   {
      System.err.println("=== Render Scene ===================================");

      // For every Position in the Scene, render the Position's Model
      // and every nested Position.
      for (Position position : scene.positionList)
      {
         System.err.println("=== Render Position ================================");
         if ( position.visible )
         {
            // Begin a (pre-order) depth-first-traversal from this Position.
            render_position_debug(scene, position, Matrix.identity(), fb);
         }
         else
         {
            System.err.println("   Hidden position.");
         }
         System.err.println("=== End Position ================================");
      }

      System.err.println("=== End Scene ===================================");
   }


   /**
      Recursively renderer a {@link Position} with quite a bit of debugging information.
      <p>
      This method does a pre-order, depth-first-traversal of the tree of
      {@link Position}'s rooted at the parameter {@code position}.
      <p>
      The pre-order "visit node" operation in this traversal first updates the
      "current transformation matrix", ({@code ctm}), using the {@link Matrix}
      in {@code position} and then recursively renders the {@link Model} in
      {@code position} using the updated {@code ctm} for the {@link Model2View} stage.

      @param scene     the {@link Scene} that we are rendering
      @param position  the current {@link Position} object to recursively render
      @param ctm       current model-to-view transformation {@link Matrix}
      @param fb        {@link FrameBuffer} to hold rendered image of the {@link Position}
   */
   private static void render_position_debug(Scene scene, Position position, Matrix ctm, FrameBuffer fb)
   {
      // Update the current model-to-view transformation matrix.
      ctm = ctm.times( position.modelMatrix );

      System.err.println("=== Render Model ===================================");

      // Render the Position's Model (if it exits and is visible).
      if ( position.model != null && position.model.visible )
      {
         // Do a pre-order, depth-first-traversal from this Model.
         render_model_debug(scene, position.model, ctm, fb);
      }
      else
      {
         System.err.println("   Hidden model.");
      }

      System.err.println("=== End Model ===================================");

      // Recursively render every nested Position of this Position.
      for (Position p : position.nestedPositions)
      {
         System.err.println("====== Render Nested Position ========================");
         if ( p.visible )
         {
            // Do a (pre-order) depth-first-traversal from this nested Position.
            render_position_debug(scene, p, ctm, fb);
         }
         else
         {
            System.err.println("   Hidden position.");
         }
         System.err.println("====== End Nested Position ===========================");
      }
   }


   /**
      Recursively renderer a {@link Model} with quite a bit of debugging information.
      <p>
      This method does a pre-order, depth-first-traversal of the tree of
      {@link Model}'s rooted at the parameter {@code model}.
      <p>
      The pre-order "visit node" operation in this traversal first updates the
      "current transformation matrix", ({@code ctm}), using the {@link Matrix}
      in {@code model} and then recursively renders {@code model} using the
      updated {@code ctm}.

      @param scene   the {@link Scene} that we are rendering
      @param model   the current {@link Model} object to recursively render
      @param ctm     current model-to-view transformation {@link Matrix}
      @param fb      {@link FrameBuffer} to hold rendered image of the {@link Position}
   */
   private static void render_model_debug(Scene scene, Model model, Matrix ctm, FrameBuffer fb)
   {
      if (model.vertexList.isEmpty() && ! model.primitiveList.isEmpty())
      {
         System.err.println("***WARNING: This model does not have any vertices.");
         System.err.println(model);
      }
      if (! model.vertexList.isEmpty() && model.primitiveList.isEmpty())
      {
         System.err.println("***WARNING: This model does not have any line segments.");
         System.err.println(model);
      }
      if (! model.vertexList.isEmpty() && model.colorList.isEmpty())
      {
         System.err.println("***WARNING: This model does not have any colors.");
         System.err.println(model);
      }

      // Update the current model-to-view transformation matrix.
      ctm = ctm.times( model.nestedMatrix );

      System.err.println("=== Render Model ===================================");

      // 0. Make a deep copy of the Model.
      Model model2 = new Model(model);

      // log interesting information
      logVertexList("0. Model    ", model2.vertexList);

      // 1. Apply the current model-to-world coordinate transformation.
      Model2World.model2world(model2.vertexList, ctm);

      // log interesting information
      logVertexList("1. World    ", model2.vertexList);

      // 2. Apply the Camera's world-to-view coordinate transformation.
      World2View.world2view(model2.vertexList, scene.camera);

      // log interesting information
      logVertexList("2. View     ", model2.vertexList);

      // 3. Apply the Camera's normalizing view-to-camera coordinate transformation.
      View2Camera.view2camera(model2.vertexList, scene.camera);

      // log interesting information
      logVertexList("3. Camera   ", model2.vertexList);

      // 4. Apply the Camera's projection transformation to homogeneous clip coordinates.
      Projection.project(model2.vertexList, scene.camera);

      // log interesting information
      logVertexList("4. Projected", model2.vertexList);
      logPrimitiveList("4. Projected", model2.primitiveList);

      // 5. Do the back face culling (of faces) operation.
      if (model2.doBackFaceCulling || model2.facesHaveTwoSides)
      {
         BackFaceCulling1.cull(model2);
      }

      // log interesting information
      logPrimitiveList("5. Culling_1", model2.primitiveList);

      // 6. Do the primitive assembly operation.
      PrimitiveAssembly.assemble(model2);

      // log interesting information
      logPrimitiveList("6. Primitive Assembly", model2.primitiveList);

      // 7. Do the back face culling (of triangles) operation.
      if (model2.doBackFaceCulling || model2.facesHaveTwoSides)
      {
         BackFaceCulling2.cull(model2);
      }

      // log interesting information
      logPrimitiveList("7. Culling_2", model2.primitiveList);

      // 8. Clip each primitive (in homogeneous clip space).
      List<Primitive> primitiveList2 = new ArrayList<>();
      for (Primitive p : model2.primitiveList)
      {
         // log interesting information
         logPrimitive("8. Clipping", model2, p);

         List<Primitive> clipped = Clip.clip(model2, p);
         if ( 0 != clipped.size() )
         {
            for (Primitive p2 : clipped)
            {
               // log interesting information
               logPrimitive("8. Clipping (accept)", model2, p2);

               // Keep the primitives that are visible.
               primitiveList2.add(p2);
            }
         }
         else
         {
            // log interesting information
            logPrimitive("8. Clipping (reject)", model2, p);
         }
      }
      model2.primitiveList = primitiveList2;

      // log interesting information
      logVertexList("8. Clipped  ", model2.vertexList);
      logPrimitiveList("8. Clipped  ", model2.primitiveList);

      // 9. Apply the perspective division operation to NDC.
      PerspectiveDivision.perspectiveDivide(model2.vertexList);

      // log interesting information
      logPrimitiveList("9. NDC", model2.primitiveList);

      // 10. Rasterize each visible primitive into pixels.
      for (Primitive p : model2.primitiveList)
      {
         // log interesting information
         logPrimitive("10. Rasterize", model2, p);

         Rasterize.rasterize(model2, p, fb);
      }
      System.err.println();


      // Recursively render every nested Model of this Model.
      if (! model2.nestedModels.isEmpty())
      {
         for (Model m : model2.nestedModels)
         {
            System.err.println("====== Render Nested Model ========================");
            if ( m.visible )
            {
               // Do a pre-order, depth-first-traversal from this nested Model.
               render_model_debug(scene, m, ctm, fb);
            }
            else
            {
               System.err.println("   Hidden position.");
            }
            System.err.println("====== End Nested Model ===========================");
         }
      }

      System.err.println("=== End Model ===================================");
   }


   /**
      This method prints a {@link String} representation of the given
      {@link Vertex} list.

      @param stage       name for the pipeline stage
      @param vertexList  vertex list whose string representation is to be printed
   */
   private static void logVertexList(String stage, List<Vertex> vertexList)
   {
      int i = 0;
      for (Vertex v : vertexList)
      {
         System.err.printf( stage + ": index = %3d, " + v.toString(), i );
         ++i;
      }
   }


   /**
      This method prints a {@link String} representation of the given
      {@link Primitive} list.

      @param stage          name for the pipeline stage
      @param primitiveList  primitive list whose string representation is to be printed
   */
   private static void logPrimitiveList(String stage, List<Primitive> primitiveList)
   {
      for (Primitive p : primitiveList)
      {
         System.err.printf( stage + ": " + p.toString() );
      }
   }


   /**
      This method prints a {@link String} representation of the given
      {@link Primitive}.

      @param stage  name for the pipeline stage
      @param model  {@link Model} that the {@link Primitive} {@code p} comes from
      @param p      {@link Primitive} whose string representation is to be printed
   */
   private static void logPrimitive(String stage, Model model, Primitive p)
   {
      System.err.print( stage + ": " + p.toString() );
      for (int vIndex : p.vIndexList)
      {
         Vertex v = model.vertexList.get(vIndex);
         System.err.printf("   vIndex = %3d, " + v.toString(), vIndex);
      }
      for (int cIndex : p.cIndexList)
      {
         Color c =  model.colorList.get(cIndex);
         System.err.printf("   cIndex = %3d, " + c.toString() + "\n", cIndex);
      }
   }
}
